/*
 * Copyright (C) 2004, 2005, 2006, 2008 Nikolas Zimmermann <zimmermann@kde.org>
 * Copyright (C) 2004, 2005, 2006 Rob Buis <buis@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef SVGLength_h
#define SVGLength_h

#include "core/css/CSSPrimitiveValue.h"
#include "core/svg/SVGLengthContext.h"
#include "core/svg/SVGParsingError.h"
#include "core/svg/properties/SVGProperty.h"
#include "platform/heap/Handle.h"

namespace blink {

class QualifiedName;

class SVGLengthTearOff;

class SVGLength final : public SVGPropertyBase {
 public:
  typedef SVGLengthTearOff TearOffType;

  static SVGLength* Create(SVGLengthMode mode = SVGLengthMode::kOther) {
    return new SVGLength(mode);
  }

  virtual void Trace(blink::Visitor*);

  SVGLength* Clone() const;
  SVGPropertyBase* CloneForAnimation(const String&) const override;

  CSSPrimitiveValue::UnitType TypeWithCalcResolved() const {
    return value_->TypeWithCalcResolved();
  }
  void SetUnitType(CSSPrimitiveValue::UnitType);
  SVGLengthMode UnitMode() const {
    return static_cast<SVGLengthMode>(unit_mode_);
  }

  bool operator==(const SVGLength&) const;
  bool operator!=(const SVGLength& other) const { return !operator==(other); }

  float Value(const SVGLengthContext&) const;
  void SetValue(float, const SVGLengthContext&);
  void SetValueAsNumber(float);

  float ValueInSpecifiedUnits() const { return value_->GetFloatValue(); }
  void SetValueInSpecifiedUnits(float value) {
    value_ = CSSPrimitiveValue::Create(value, value_->TypeWithCalcResolved());
  }

  const CSSPrimitiveValue& AsCSSPrimitiveValue() const { return *value_; }

  // Resolves LengthTypePercentage into a normalized floating point number (full
  // value is 1.0).
  float ValueAsPercentage() const;

  // Returns a number to be used as percentage (so full value is 100)
  float ValueAsPercentage100() const;

  // Scale the input value by this SVGLength. Higher precision than input *
  // valueAsPercentage().
  float ScaleByPercentage(float) const;

  String ValueAsString() const override;
  SVGParsingError SetValueAsString(const String&);

  void NewValueSpecifiedUnits(CSSPrimitiveValue::UnitType,
                              float value_in_specified_units);
  void ConvertToSpecifiedUnits(CSSPrimitiveValue::UnitType,
                               const SVGLengthContext&);

  // Helper functions
  bool IsRelative() const {
    return CSSPrimitiveValue::IsRelativeUnit(value_->TypeWithCalcResolved());
  }
  bool IsFontRelative() const { return value_->IsFontRelativeLength(); }
  bool IsCalculated() const { return value_->IsCalculated(); }

  bool IsZero() const { return value_->GetFloatValue() == 0; }

  static SVGLengthMode LengthModeForAnimatedLengthAttribute(
      const QualifiedName&);
  static bool NegativeValuesForbiddenForAnimatedLengthAttribute(
      const QualifiedName&);

  void Add(SVGPropertyBase*, SVGElement*) override;
  void CalculateAnimatedValue(SVGAnimationElement*,
                              float percentage,
                              unsigned repeat_count,
                              SVGPropertyBase* from,
                              SVGPropertyBase* to,
                              SVGPropertyBase* to_at_end_of_duration_value,
                              SVGElement* context_element) override;
  float CalculateDistance(SVGPropertyBase* to,
                          SVGElement* context_element) override;

  static AnimatedPropertyType ClassType() { return kAnimatedLength; }
  AnimatedPropertyType GetType() const override { return ClassType(); }

 private:
  SVGLength(SVGLengthMode);
  SVGLength(const SVGLength&);

  Member<const CSSPrimitiveValue> value_;
  unsigned unit_mode_ : 2;
};

DEFINE_SVG_PROPERTY_TYPE_CASTS(SVGLength);

}  // namespace blink

#endif  // SVGLength_h
