// Copyright 2017 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_UI_WEBUI_PRINT_PREVIEW_PDF_PRINTER_HANDLER_H_
#define CHROME_BROWSER_UI_WEBUI_PRINT_PREVIEW_PDF_PRINTER_HANDLER_H_

#include <memory>
#include <string>

#include "base/callback.h"
#include "base/files/file_path.h"
#include "base/memory/ref_counted.h"
#include "base/memory/weak_ptr.h"
#include "base/strings/string16.h"
#include "chrome/browser/ui/webui/print_preview/printer_handler.h"
#include "ui/shell_dialogs/select_file_dialog.h"

namespace base {
class RefCountedBytes;
}

namespace content {
class WebContents;
}

namespace gfx {
class Size;
}

namespace printing {
class StickySettings;
}

class GURL;
class Profile;

class PdfPrinterHandler : public PrinterHandler,
                          public ui::SelectFileDialog::Listener {
 public:
  PdfPrinterHandler(Profile* profile,
                    content::WebContents* preview_web_contents,
                    printing::StickySettings* sticky_settings);

  ~PdfPrinterHandler() override;

  // PrinterHandler implementation
  void Reset() override;
  // Required by PrinterHandler implementation but should never be called.
  void StartGetPrinters(const AddedPrintersCallback& added_printers_callback,
                        GetPrintersDoneCallback done_callback) override;
  void StartGetCapability(const std::string& destination_id,
                          GetCapabilityCallback callback) override;
  void StartPrint(const std::string& destination_id,
                  const std::string& capability,
                  const base::string16& job_title,
                  const std::string& ticket_json,
                  const gfx::Size& page_size,
                  const scoped_refptr<base::RefCountedBytes>& print_data,
                  PrintCallback callback) override;

  // SelectFileDialog::Listener implementation.
  void FileSelected(const base::FilePath& path,
                    int index,
                    void* params) override;
  void FileSelectionCanceled(void* params) override;

  // Sets |pdf_file_saved_closure_| to |closure|.
  void SetPdfSavedClosureForTesting(const base::Closure& closure);

  // Exposed for testing.
  static base::FilePath GetFileNameForPrintJobTitle(
      const base::string16& job_title);
  static base::FilePath GetFileNameForURL(const GURL& url);
  static base::FilePath GetFileName(const GURL& url,
                                    const base::string16& job_title,
                                    bool is_savable);

 protected:
  virtual void SelectFile(const base::FilePath& default_filename,
                          content::WebContents* initiator,
                          bool prompt_user);

  // The print preview web contents. Protected so unit tests can access it.
  content::WebContents* const preview_web_contents_;

  // The underlying dialog object. Protected so unit tests can access it.
  scoped_refptr<ui::SelectFileDialog> select_file_dialog_;

 private:
  void PostPrintToPdfTask();
  void OnGotUniqueFileName(const base::FilePath& path);
  void OnDirectoryCreated(const base::FilePath& path);

  Profile* const profile_;
  printing::StickySettings* const sticky_settings_;

  // Holds the path to the print to pdf request. It is empty if no such request
  // exists.
  base::FilePath print_to_pdf_path_;

  // Notifies tests that want to know if the PDF has been saved. This doesn't
  // notify the test if it was a successful save, only that it was attempted.
  base::Closure pdf_file_saved_closure_;

  // The data to print
  scoped_refptr<base::RefCountedBytes> print_data_;

  // The callback to call when complete.
  PrintCallback print_callback_;

  base::WeakPtrFactory<PdfPrinterHandler> weak_ptr_factory_;

  DISALLOW_COPY_AND_ASSIGN(PdfPrinterHandler);
};

#endif  // CHROME_BROWSER_UI_WEBUI_PRINT_PREVIEW_PDF_PRINTER_HANDLER_H_
