// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef WebGestureEvent_h
#define WebGestureEvent_h

#include "WebFloatSize.h"
#include "WebGestureDevice.h"
#include "WebInputEvent.h"

namespace blink {

// See WebInputEvent.h for details why this pack is here.
#pragma pack(push, 4)

// WebGestureEvent ---------------------------------------------------------

class WebGestureEvent : public WebInputEvent {
 public:
  enum ScrollUnits {
    kPrecisePixels = 0,  // generated by high precision devices.
    kPixels,             // large pixel jump duration; should animate to delta.
    kPage,               // page (visible viewport) based scrolling.
    kLastScrollUnit = kPage,
  };

  enum InertialPhaseState {
    kUnknownMomentumPhase = 0,  // No phase information.
    kNonMomentumPhase,          // Regular scrolling phase.
    kMomentumPhase,             // Momentum phase.
    kLastPhase = kMomentumPhase,
  };

  // TODO(mustaq): Make these coordinates private & fractional, as in
  // WebMouseEvent.h .
  int x;
  int y;
  int global_x;
  int global_y;
  WebGestureDevice source_device;
  bool is_source_touch_event_set_non_blocking;

  // The pointer type for the first touch point in the gesture.
  WebPointerProperties::PointerType primary_pointer_type =
      WebPointerProperties::PointerType::kUnknown;

  // If the WebGestureEvent has sourceDevice=WebGestureDeviceTouchscreen, this
  // field contains the unique identifier for the touch event that released
  // this event at TouchDispositionGestureFilter. If the WebGestureEvents was
  // not released through a touch event (e.g. timer-released gesture events or
  // gesture events with sourceDevice!=WebGestureDeviceTouchscreen), the field
  // contains 0. See crbug.com/618738.
  uint32_t unique_touch_event_id;

  // This field exists to allow BrowserPlugin to mark GestureScroll events as
  // 'resent' to handle the case where an event is not consumed when first
  // encountered; it should be handled differently by the plugin when it is
  // sent for thesecond time. No code within Blink touches this, other than to
  // plumb it through event conversions.
  int resending_plugin_id;

  union {
    // Tap information must be set for GestureTap, GestureTapUnconfirmed,
    // and GestureDoubleTap events.
    struct {
      int tap_count;
      float width;
      float height;
    } tap;

    struct {
      float width;
      float height;
    } tap_down;

    struct {
      float width;
      float height;
    } show_press;

    struct {
      float width;
      float height;
    } long_press;

    struct {
      float first_finger_width;
      float first_finger_height;
    } two_finger_tap;

    struct {
      // Initial motion that triggered the scroll.
      float delta_x_hint;
      float delta_y_hint;
      // Default initialized to ScrollUnits::PrecisePixels.
      ScrollUnits delta_hint_units;
      // If true, this event will skip hit testing to find a scroll
      // target and instead just scroll the viewport.
      bool target_viewport;
      // The state of inertial phase scrolling. OSX has unique phases for normal
      // and momentum scroll events. Should always be UnknownMomentumPhase for
      // touch based input as it generates GestureFlingStart instead.
      InertialPhaseState inertial_phase;
      // True if this event was synthesized in order to force a hit test;
      // avoiding scroll latching behavior until crbug.com/526463 is fully
      // implemented.
      bool synthetic;

      // number of pointers down.
      int pointer_count;
    } scroll_begin;

    struct {
      float delta_x;
      float delta_y;
      float velocity_x;
      float velocity_y;
      // Whether any previous GestureScrollUpdate in the current scroll
      // sequence was suppressed (e.g., the causal touchmove was
      // preventDefault'ed). This bit is particularly useful for
      // determining whether the observed scroll update sequence captures
      // the entirety of the generative motion.
      bool previous_update_in_sequence_prevented;
      bool prevent_propagation;
      InertialPhaseState inertial_phase;
      // Default initialized to ScrollUnits::PrecisePixels.
      ScrollUnits delta_units;
    } scroll_update;

    struct {
      // The original delta units the scrollBegin and scrollUpdates
      // were sent as.
      ScrollUnits delta_units;
      // The state of inertial phase scrolling. OSX has unique phases for normal
      // and momentum scroll events. Should always be UnknownMomentumPhase for
      // touch based input as it generates GestureFlingStart instead.
      InertialPhaseState inertial_phase;
      // True if this event was synthesized in order to generate the proper
      // GSB/GSU/GSE matching sequences. This is a temporary so that a future
      // GSB will generate a hit test so latching behavior is avoided
      // until crbug.com/526463 is fully implemented.
      bool synthetic;
    } scroll_end;

    struct {
      float velocity_x;
      float velocity_y;
      // If true, this event will skip hit testing to find a scroll
      // target and instead just scroll the viewport.
      bool target_viewport;
    } fling_start;

    struct {
      bool target_viewport;
      // If set to true, don't treat flingCancel
      // as a part of fling boost events sequence.
      bool prevent_boosting;
    } fling_cancel;

    struct {
      bool zoom_disabled;
      float scale;
    } pinch_update;
  } data;

  WebGestureEvent(Type type, int modifiers, double time_stamp_seconds)
      : WebInputEvent(sizeof(WebGestureEvent),
                      type,
                      modifiers,
                      time_stamp_seconds),
        source_device(kWebGestureDeviceUninitialized),
        resending_plugin_id(-1) {}

  WebGestureEvent()
      : WebInputEvent(sizeof(WebGestureEvent)),
        source_device(kWebGestureDeviceUninitialized),
        resending_plugin_id(-1) {}

  WebFloatPoint PositionInWidget() const { return WebFloatPoint(x, y); }
  WebFloatPoint PositionInScreen() const {
    return WebFloatPoint(global_x, global_y);
  }

#if INSIDE_BLINK
  BLINK_PLATFORM_EXPORT float DeltaXInRootFrame() const;
  BLINK_PLATFORM_EXPORT float DeltaYInRootFrame() const;
  BLINK_PLATFORM_EXPORT ScrollUnits DeltaUnits() const;
  BLINK_PLATFORM_EXPORT WebFloatPoint PositionInRootFrame() const;
  BLINK_PLATFORM_EXPORT float PinchScale() const;
  BLINK_PLATFORM_EXPORT InertialPhaseState InertialPhase() const;
  BLINK_PLATFORM_EXPORT bool Synthetic() const;

  BLINK_PLATFORM_EXPORT float VelocityX() const;
  BLINK_PLATFORM_EXPORT float VelocityY() const;

  BLINK_PLATFORM_EXPORT WebFloatSize TapAreaInRootFrame() const;
  BLINK_PLATFORM_EXPORT int TapCount() const;

  BLINK_PLATFORM_EXPORT void ApplyTouchAdjustment(
      WebFloatPoint root_frame_coords);

  // Sets any scaled values to be their computed values and sets |frameScale|
  // back to 1 and |translateX|, |translateY| back to 0.
  BLINK_PLATFORM_EXPORT void FlattenTransform();

  bool PreventPropagation() const {
    // TODO(tdresser) Once we've decided if we're getting rid of scroll
    // chaining, we should remove all scroll chaining related logic. See
    // crbug.com/526462 for details.
    DCHECK(type_ == WebInputEvent::kGestureScrollUpdate);
    return true;
  }

  bool IsScrollEvent() const {
    switch (type_) {
      case kGestureScrollBegin:
      case kGestureScrollEnd:
      case kGestureScrollUpdate:
      case kGestureFlingStart:
      case kGestureFlingCancel:
      case kGesturePinchBegin:
      case kGesturePinchEnd:
      case kGesturePinchUpdate:
        return true;
      case kGestureTap:
      case kGestureTapUnconfirmed:
      case kGestureTapDown:
      case kGestureShowPress:
      case kGestureTapCancel:
      case kGestureTwoFingerTap:
      case kGestureLongPress:
      case kGestureLongTap:
        return false;
      default:
        NOTREACHED();
        return false;
    }
  }
#endif

  bool IsTargetViewport() const {
    switch (type_) {
      case kGestureScrollBegin:
        return data.scroll_begin.target_viewport;
      case kGestureFlingStart:
        return data.fling_start.target_viewport;
      case kGestureFlingCancel:
        return data.fling_cancel.target_viewport;
      default:
        return false;
    }
  }
};

#pragma pack(pop)

}  // namespace blink

#endif  // WebGestureEvent_h
