/*
 * Copyright (C) 2006, 2007, 2008, 2009, 2010 Apple Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 */

#ifndef RefCounted_h
#define RefCounted_h

#include "platform/wtf/Allocator.h"
#include "platform/wtf/Assertions.h"
#include "platform/wtf/Noncopyable.h"
#include "platform/wtf/WTFExport.h"

#if DCHECK_IS_ON()
#define CHECK_REF_COUNTED_LIFECYCLE 1
#include "platform/wtf/ThreadRestrictionVerifier.h"
#else
#define CHECK_REF_COUNTED_LIFECYCLE 0
#endif

namespace WTF {

// This base class holds the non-template methods and attributes.
// The RefCounted class inherits from it reducing the template bloat
// generated by the compiler (technique called template hoisting).
class WTF_EXPORT RefCountedBase {
 public:
  void Ref() const {
#if CHECK_REF_COUNTED_LIFECYCLE
    SECURITY_DCHECK(verifier_.OnRef(ref_count_));
    DCHECK(!adoption_is_required_);
#endif
    SECURITY_DCHECK(!deletion_has_begun_);
    ++ref_count_;
  }

  bool HasOneRef() const {
    SECURITY_DCHECK(!deletion_has_begun_);
#if CHECK_REF_COUNTED_LIFECYCLE
    SECURITY_DCHECK(verifier_.IsSafeToUse());
#endif
    return ref_count_ == 1;
  }

  int RefCount() const {
#if CHECK_REF_COUNTED_LIFECYCLE
    SECURITY_DCHECK(verifier_.IsSafeToUse());
#endif
    return ref_count_;
  }

 protected:
  RefCountedBase()
      : ref_count_(1)
#if ENABLE_SECURITY_ASSERT
        ,
        deletion_has_begun_(false)
#endif
#if CHECK_REF_COUNTED_LIFECYCLE
        ,
        adoption_is_required_(true)
#endif
  {
  }

  ~RefCountedBase() {
    SECURITY_DCHECK(deletion_has_begun_);
#if CHECK_REF_COUNTED_LIFECYCLE
    DCHECK(!adoption_is_required_);
#endif
  }

  // Returns whether the pointer should be freed or not.
  bool DerefBase() const {
    SECURITY_DCHECK(!deletion_has_begun_);
#if CHECK_REF_COUNTED_LIFECYCLE
    SECURITY_DCHECK(verifier_.OnDeref(ref_count_));
    DCHECK(!adoption_is_required_);
#endif

    DCHECK_GT(ref_count_, 0);
    --ref_count_;
    if (!ref_count_) {
#if ENABLE_SECURITY_ASSERT
      deletion_has_begun_ = true;
#endif
      return true;
    }

    return false;
  }

#if CHECK_REF_COUNTED_LIFECYCLE
  bool DeletionHasBegun() const { return deletion_has_begun_; }
#endif

 private:
#if CHECK_REF_COUNTED_LIFECYCLE || ENABLE_SECURITY_ASSERT
  friend void Adopted(RefCountedBase*);
#endif

  mutable int ref_count_;
#if ENABLE_SECURITY_ASSERT
  mutable bool deletion_has_begun_;
#endif
#if CHECK_REF_COUNTED_LIFECYCLE
  mutable bool adoption_is_required_;
  mutable ThreadRestrictionVerifier verifier_;
#endif
};

#if CHECK_REF_COUNTED_LIFECYCLE || ENABLE_SECURITY_ASSERT
inline void Adopted(RefCountedBase* object) {
  if (!object)
    return;
  SECURITY_DCHECK(!object->deletion_has_begun_);
#if CHECK_REF_COUNTED_LIFECYCLE
  object->adoption_is_required_ = false;
#endif
}
#endif

template <typename T>
class RefCounted : public RefCountedBase {
  WTF_MAKE_NONCOPYABLE(RefCounted);

  // Put |T| in here instead of |RefCounted| so the heap profiler reports |T|
  // instead of |RefCounted<T>|. This does not affect overloading of operator
  // new.
  USING_FAST_MALLOC(T);

 public:
  void Deref() const {
    if (DerefBase())
      delete static_cast<const T*>(this);
  }

 protected:
  RefCounted() {}
};

// Allows subclasses to use the default copy constructor.
template <typename T>
class RefCountedCopyable : public RefCounted<T> {
 protected:
  RefCountedCopyable() = default;
  RefCountedCopyable(const RefCountedCopyable&) : RefCounted<T>() {}
};

}  // namespace WTF

using WTF::RefCounted;
using WTF::RefCountedCopyable;

#endif  // RefCounted_h
