#include <iostream>
#include <string>

#include "vsg/all.h"

#ifdef _WIN32
#    define WIN32_LEAN_AND_MEAN
#    include <windows.h>
#endif

template<class C1, class C2>
bool test(const std::basic_string<C1>& input, const std::basic_string<C2>& expectedOutput)
{
    auto output = vsg::convert_utf<std::basic_string<C2>>(input);
    bool pass = output == expectedOutput;
    std::cout << "Test case: ";
    if constexpr (std::is_same_v<C1, char>)
    {
        std::cout << input;
    }
    else if (std::is_same_v<C2, char>)
    {
        std::cout << expectedOutput;
    }
    std::cout << (pass ? " matches" : " does not match") << std::endl;
    return pass;
}

int main(int, char**)
{
#ifdef _WIN32
    // Windows is backwards-compatible with things that predate UTF-8's invention by default
    SetConsoleOutputCP(CP_UTF8);
#endif
    using namespace std::string_literals;

    bool allPassed = true;

    // spaces stop ES turning into Spanish flag
    allPassed &= test(L"testTEST 🇹 🇪 🇸 🇹 "s, u8"testTEST 🇹 🇪 🇸 🇹 "s);
    allPassed &= test(u8"testTEST 🇹 🇪 🇸 🇹 "s, L"testTEST 🇹 🇪 🇸 🇹 "s);

    // pangrams in various languages
    allPassed &= test(L"Под южно дърво, цъфтящо в синьо, бягаше малко пухкаво зайче."s, u8"Под южно дърво, цъфтящо в синьо, бягаше малко пухкаво зайче."s);
    allPassed &= test(u8"Под южно дърво, цъфтящо в синьо, бягаше малко пухкаво зайче."s, L"Под южно дърво, цъфтящо в синьо, бягаше малко пухкаво зайче."s);
    allPassed &= test(L"Příliš žluťoučký kůň úpěl ďábelské ódy."s, u8"Příliš žluťoučký kůň úpěl ďábelské ódy."s);
    allPassed &= test(u8"Příliš žluťoučký kůň úpěl ďábelské ódy."s, L"Příliš žluťoučký kůň úpěl ďábelské ódy."s);
    allPassed &= test(L"視野無限廣，窗外有藍天"s, u8"視野無限廣，窗外有藍天"s);
    allPassed &= test(u8"視野無限廣，窗外有藍天"s, L"視野無限廣，窗外有藍天"s);
    allPassed &= test(L"Quizdeltagerne spiste jordbær med fløde, mens cirkusklovnen Walther spillede på xylofon."s, u8"Quizdeltagerne spiste jordbær med fløde, mens cirkusklovnen Walther spillede på xylofon."s);
    allPassed &= test(u8"Quizdeltagerne spiste jordbær med fløde, mens cirkusklovnen Walther spillede på xylofon."s, L"Quizdeltagerne spiste jordbær med fløde, mens cirkusklovnen Walther spillede på xylofon."s);
    allPassed &= test(L"Pa’s wijze lynx bezag vroom het fikse aquaduct."s, u8"Pa’s wijze lynx bezag vroom het fikse aquaduct."s);
    allPassed &= test(u8"Pa’s wijze lynx bezag vroom het fikse aquaduct."s, L"Pa’s wijze lynx bezag vroom het fikse aquaduct."s);
    allPassed &= test(L"Eĥoŝanĝo ĉiuĵaŭde."s, u8"Eĥoŝanĝo ĉiuĵaŭde."s);
    allPassed &= test(u8"Eĥoŝanĝo ĉiuĵaŭde."s, L"Eĥoŝanĝo ĉiuĵaŭde."s);
    allPassed &= test(L"See väike mölder jõuab rongile hüpata"s, u8"See väike mölder jõuab rongile hüpata"s);
    allPassed &= test(u8"See väike mölder jõuab rongile hüpata"s, L"See väike mölder jõuab rongile hüpata"s);
    allPassed &= test(L"Viekas kettu punaturkki laiskan koiran takaa kurkki."s, u8"Viekas kettu punaturkki laiskan koiran takaa kurkki."s);
    allPassed &= test(u8"Viekas kettu punaturkki laiskan koiran takaa kurkki."s, L"Viekas kettu punaturkki laiskan koiran takaa kurkki."s);
    allPassed &= test(L"Voix ambiguë d’un cœur qui au zéphyr préfère les jattes de kiwis."s, u8"Voix ambiguë d’un cœur qui au zéphyr préfère les jattes de kiwis."s);
    allPassed &= test(u8"Voix ambiguë d’un cœur qui au zéphyr préfère les jattes de kiwis."s, L"Voix ambiguë d’un cœur qui au zéphyr préfère les jattes de kiwis."s);
    allPassed &= test(L"Zwölf Boxkämpfer jagen Viktor quer über den großen Sylter Deich"s, u8"Zwölf Boxkämpfer jagen Viktor quer über den großen Sylter Deich"s);
    allPassed &= test(u8"Zwölf Boxkämpfer jagen Viktor quer über den großen Sylter Deich"s, L"Zwölf Boxkämpfer jagen Viktor quer über den großen Sylter Deich"s);
    allPassed &= test(L"Ο καλύμνιος σφουγγαράς ψιθύρισε πως θα βουτήξει χωρίς να διστάζει."s, u8"Ο καλύμνιος σφουγγαράς ψιθύρισε πως θα βουτήξει χωρίς να διστάζει."s);
    allPassed &= test(u8"Ο καλύμνιος σφουγγαράς ψιθύρισε πως θα βουτήξει χωρίς να διστάζει."s, L"Ο καλύμνιος σφουγγαράς ψιθύρισε πως θα βουτήξει χωρίς να διστάζει."s);
    allPassed &= test(L"דג סקרן שט לו בים זך אך לפתע פגש חבורה נחמדה שצצה כך."s, u8"דג סקרן שט לו בים זך אך לפתע פגש חבורה נחמדה שצצה כך."s);
    allPassed &= test(u8"דג סקרן שט לו בים זך אך לפתע פגש חבורה נחמדה שצצה כך."s, L"דג סקרן שט לו בים זך אך לפתע פגש חבורה נחמדה שצצה כך."s);
    allPassed &= test(L"Árvíztűrő tükörfúrógép"s, u8"Árvíztűrő tükörfúrógép"s);
    allPassed &= test(u8"Árvíztűrő tükörfúrógép"s, L"Árvíztűrő tükörfúrógép"s);
    allPassed &= test(L"いろはにほへと ちりぬるを わかよたれそ つねならむ うゐのおくやま けふこえて あさきゆめみし ゑひもせす"s, u8"いろはにほへと ちりぬるを わかよたれそ つねならむ うゐのおくやま けふこえて あさきゆめみし ゑひもせす"s);
    allPassed &= test(u8"いろはにほへと ちりぬるを わかよたれそ つねならむ うゐのおくやま けふこえて あさきゆめみし ゑひもせす"s, L"いろはにほへと ちりぬるを わかよたれそ つねならむ うゐのおくやま けふこえて あさきゆめみし ゑひもせす"s);
    allPassed &= test(L"다람쥐 헌 쳇바퀴에 타고파"s, u8"다람쥐 헌 쳇바퀴에 타고파"s);
    allPassed &= test(u8"다람쥐 헌 쳇바퀴에 타고파"s, L"다람쥐 헌 쳇바퀴에 타고파"s);
    allPassed &= test(L"Sarkanās jūrascūciņas peld pa jūru."s, u8"Sarkanās jūrascūciņas peld pa jūru."s);
    allPassed &= test(u8"Sarkanās jūrascūciņas peld pa jūru."s, L"Sarkanās jūrascūciņas peld pa jūru."s);
    allPassed &= test(L"En god stil må først og fremst være klar. Den må være passende. Aristoteles."s, u8"En god stil må først og fremst være klar. Den må være passende. Aristoteles."s);
    allPassed &= test(u8"En god stil må først og fremst være klar. Den må være passende. Aristoteles."s, L"En god stil må først og fremst være klar. Den må være passende. Aristoteles."s);
    allPassed &= test(L"Pchnąć w tę łódź jeża lub ośm skrzyń fig"s, u8"Pchnąć w tę łódź jeża lub ośm skrzyń fig"s);
    allPassed &= test(u8"Pchnąć w tę łódź jeża lub ośm skrzyń fig"s, L"Pchnąć w tę łódź jeża lub ośm skrzyń fig"s);
    allPassed &= test(L"Съешь ещё этих мягких французских булок да выпей же чаю"s, u8"Съешь ещё этих мягких французских булок да выпей же чаю"s);
    allPassed &= test(u8"Съешь ещё этих мягких французских булок да выпей же чаю"s, L"Съешь ещё этих мягких французских булок да выпей же чаю"s);
    allPassed &= test(L"Чешће цeђење мрeжастим џаком побољшава фертилизацију генских хибрида."s, u8"Чешће цeђење мрeжастим џаком побољшава фертилизацију генских хибрида."s);
    allPassed &= test(u8"Чешће цeђење мрeжастим џаком побољшава фертилизацију генских хибрида."s, L"Чешће цeђење мрeжастим џаком побољшава фертилизацију генских хибрида."s);
    allPassed &= test(L"Češće ceđenje mrežastim džakom poboljšava fertilizaciju genskih hibrida."s, u8"Češće ceđenje mrežastim džakom poboljšava fertilizaciju genskih hibrida."s);
    allPassed &= test(u8"Češće ceđenje mrežastim džakom poboljšava fertilizaciju genskih hibrida."s, L"Češće ceđenje mrežastim džakom poboljšava fertilizaciju genskih hibrida."s);
    allPassed &= test(L"Kŕdeľ šťastných ďatľov učí pri ústí Váhu mĺkveho koňa obhrýzať kôru a žrať čerstvé mäso."s, u8"Kŕdeľ šťastných ďatľov učí pri ústí Váhu mĺkveho koňa obhrýzať kôru a žrať čerstvé mäso."s);
    allPassed &= test(u8"Kŕdeľ šťastných ďatľov učí pri ústí Váhu mĺkveho koňa obhrýzať kôru a žrať čerstvé mäso."s, L"Kŕdeľ šťastných ďatľov učí pri ústí Váhu mĺkveho koňa obhrýzať kôru a žrať čerstvé mäso."s);
    allPassed &= test(L"เป็นมนุษย์สุดประเสริฐเลิศคุณค่ากว่าบรรดาฝูงสัตว์เดรัจฉานจงฝ่าฟันพัฒนาวิชาการ อย่าล้างผลาญฤๅเข่นฆ่าบีฑาใครไม่ถือโทษโกรธแช่งซัดฮึดฮัดด่าหัดอภัยเหมือนกีฬาอัชฌาสัยปฏิบัติประพฤติกฎกำหนดใจพูดจาให้จ๊ะ ๆ จ๋า ๆ น่าฟังเอยฯ"s, u8"เป็นมนุษย์สุดประเสริฐเลิศคุณค่ากว่าบรรดาฝูงสัตว์เดรัจฉานจงฝ่าฟันพัฒนาวิชาการ อย่าล้างผลาญฤๅเข่นฆ่าบีฑาใครไม่ถือโทษโกรธแช่งซัดฮึดฮัดด่าหัดอภัยเหมือนกีฬาอัชฌาสัยปฏิบัติประพฤติกฎกำหนดใจพูดจาให้จ๊ะ ๆ จ๋า ๆ น่าฟังเอยฯ"s);
    allPassed &= test(u8"เป็นมนุษย์สุดประเสริฐเลิศคุณค่ากว่าบรรดาฝูงสัตว์เดรัจฉานจงฝ่าฟันพัฒนาวิชาการ อย่าล้างผลาญฤๅเข่นฆ่าบีฑาใครไม่ถือโทษโกรธแช่งซัดฮึดฮัดด่าหัดอภัยเหมือนกีฬาอัชฌาสัยปฏิบัติประพฤติกฎกำหนดใจพูดจาให้จ๊ะ ๆ จ๋า ๆ น่าฟังเอยฯ"s, L"เป็นมนุษย์สุดประเสริฐเลิศคุณค่ากว่าบรรดาฝูงสัตว์เดรัจฉานจงฝ่าฟันพัฒนาวิชาการ อย่าล้างผลาญฤๅเข่นฆ่าบีฑาใครไม่ถือโทษโกรธแช่งซัดฮึดฮัดด่าหัดอภัยเหมือนกีฬาอัชฌาสัยปฏิบัติประพฤติกฎกำหนดใจพูดจาให้จ๊ะ ๆ จ๋า ๆ น่าฟังเอยฯ"s);
    allPassed &= test(L"زۆھرەگۈل ئابدۇۋاجىت فرانسىيەنىڭ پارىژدىكى خېلى بىشەم ئوقۇغۇچى."s, u8"زۆھرەگۈل ئابدۇۋاجىت فرانسىيەنىڭ پارىژدىكى خېلى بىشەم ئوقۇغۇچى."s);
    allPassed &= test(u8"زۆھرەگۈل ئابدۇۋاجىت فرانسىيەنىڭ پارىژدىكى خېلى بىشەم ئوقۇغۇچى."s, L"زۆھرەگۈل ئابدۇۋاجىت فرانسىيەنىڭ پارىژدىكى خېلى بىشەم ئوقۇغۇچى."s);

    //symbols
    allPassed &= test(L"∮ E⋅da = Q,  n → ∞, ∑ f(i) = ∏ g(i), ∀x∈ℝ: ⌈x⌉ = −⌊−x⌋, α ∧ ¬β = ¬(¬α ∨ β)"s, u8"∮ E⋅da = Q,  n → ∞, ∑ f(i) = ∏ g(i), ∀x∈ℝ: ⌈x⌉ = −⌊−x⌋, α ∧ ¬β = ¬(¬α ∨ β)"s);
    allPassed &= test(u8"∮ E⋅da = Q,  n → ∞, ∑ f(i) = ∏ g(i), ∀x∈ℝ: ⌈x⌉ = −⌊−x⌋, α ∧ ¬β = ¬(¬α ∨ β)"s, L"∮ E⋅da = Q,  n → ∞, ∑ f(i) = ∏ g(i), ∀x∈ℝ: ⌈x⌉ = −⌊−x⌋, α ∧ ¬β = ¬(¬α ∨ β)"s);
    allPassed &= test(L"ℕ ⊆ ℕ₀ ⊂ ℤ ⊂ ℚ ⊂ ℝ ⊂ ℂ, ⊥ < a ≠ b ≡ c ≤ d ≪ ⊤ ⇒ (A ⇔ B)"s, u8"ℕ ⊆ ℕ₀ ⊂ ℤ ⊂ ℚ ⊂ ℝ ⊂ ℂ, ⊥ < a ≠ b ≡ c ≤ d ≪ ⊤ ⇒ (A ⇔ B)"s);
    allPassed &= test(u8"ℕ ⊆ ℕ₀ ⊂ ℤ ⊂ ℚ ⊂ ℝ ⊂ ℂ, ⊥ < a ≠ b ≡ c ≤ d ≪ ⊤ ⇒ (A ⇔ B)"s, L"ℕ ⊆ ℕ₀ ⊂ ℤ ⊂ ℚ ⊂ ℝ ⊂ ℂ, ⊥ < a ≠ b ≡ c ≤ d ≪ ⊤ ⇒ (A ⇔ B)"s);
    allPassed &= test(L"2H₂ + O₂ ⇌ 2H₂O, R = 4.7 kΩ, ⌀ 200 mm"s, u8"2H₂ + O₂ ⇌ 2H₂O, R = 4.7 kΩ, ⌀ 200 mm"s);
    allPassed &= test(u8"2H₂ + O₂ ⇌ 2H₂O, R = 4.7 kΩ, ⌀ 200 mm"s, L"2H₂ + O₂ ⇌ 2H₂O, R = 4.7 kΩ, ⌀ 200 mm"s);
    allPassed &= test(L"((V⍳V)=⍳⍴V)/V←,V    ⌷←⍳→⍴∆∇⊃‾⍎⍕⌈"s, u8"((V⍳V)=⍳⍴V)/V←,V    ⌷←⍳→⍴∆∇⊃‾⍎⍕⌈"s);
    allPassed &= test(u8"((V⍳V)=⍳⍴V)/V←,V    ⌷←⍳→⍴∆∇⊃‾⍎⍕⌈"s, L"((V⍳V)=⍳⍴V)/V←,V    ⌷←⍳→⍴∆∇⊃‾⍎⍕⌈"s);

    // Supplementary Multilingual Plane
    allPassed &= test(L"𐊀𐊁𐊂𐊃𐊄𐊅𐊆𐊇𐊈𐊉𐊊𐊋𐊌𐊍𐊎𐊏𐊐𐊑𐊒𐊓𐊔𐊕𐊖𐊗𐊘𐊙𐊚𐊛𐊜"s, u8"𐊀𐊁𐊂𐊃𐊄𐊅𐊆𐊇𐊈𐊉𐊊𐊋𐊌𐊍𐊎𐊏𐊐𐊑𐊒𐊓𐊔𐊕𐊖𐊗𐊘𐊙𐊚𐊛𐊜"s);
    allPassed &= test(u8"𐊀𐊁𐊂𐊃𐊄𐊅𐊆𐊇𐊈𐊉𐊊𐊋𐊌𐊍𐊎𐊏𐊐𐊑𐊒𐊓𐊔𐊕𐊖𐊗𐊘𐊙𐊚𐊛𐊜"s, L"𐊀𐊁𐊂𐊃𐊄𐊅𐊆𐊇𐊈𐊉𐊊𐊋𐊌𐊍𐊎𐊏𐊐𐊑𐊒𐊓𐊔𐊕𐊖𐊗𐊘𐊙𐊚𐊛𐊜"s);
    allPassed &= test(L"🀰🀱🀲🀳🀴🀵🀶🀷🀸🀹🀺🀻🀼🀽🀾🀿🁀🁁🁂🁃🁄🁅🁆🁇🁈🁉🁊🁋🁌🁍🁎🁏🁐🁑🁒🁓🁔🁕🁖🁗🁘🁙🁚🁛🁜🁝🁞🁟🁠🁡🁢🁣🁤🁥🁦🁧🁨🁩🁪🁫🁬🁭🁮🁯🁰🁱🁲🁳🁴🁵🁶🁷🁸🁹🁺🁻🁼🁽🁾🁿🂀🂁🂂🂃🂄🂅🂆🂇🂈🂉🂊🂋🂌🂍🂎🂏🂐🂑🂒🂓"s, u8"🀰🀱🀲🀳🀴🀵🀶🀷🀸🀹🀺🀻🀼🀽🀾🀿🁀🁁🁂🁃🁄🁅🁆🁇🁈🁉🁊🁋🁌🁍🁎🁏🁐🁑🁒🁓🁔🁕🁖🁗🁘🁙🁚🁛🁜🁝🁞🁟🁠🁡🁢🁣🁤🁥🁦🁧🁨🁩🁪🁫🁬🁭🁮🁯🁰🁱🁲🁳🁴🁵🁶🁷🁸🁹🁺🁻🁼🁽🁾🁿🂀🂁🂂🂃🂄🂅🂆🂇🂈🂉🂊🂋🂌🂍🂎🂏🂐🂑🂒🂓"s);
    allPassed &= test(u8"🀰🀱🀲🀳🀴🀵🀶🀷🀸🀹🀺🀻🀼🀽🀾🀿🁀🁁🁂🁃🁄🁅🁆🁇🁈🁉🁊🁋🁌🁍🁎🁏🁐🁑🁒🁓🁔🁕🁖🁗🁘🁙🁚🁛🁜🁝🁞🁟🁠🁡🁢🁣🁤🁥🁦🁧🁨🁩🁪🁫🁬🁭🁮🁯🁰🁱🁲🁳🁴🁵🁶🁷🁸🁹🁺🁻🁼🁽🁾🁿🂀🂁🂂🂃🂄🂅🂆🂇🂈🂉🂊🂋🂌🂍🂎🂏🂐🂑🂒🂓"s, L"🀰🀱🀲🀳🀴🀵🀶🀷🀸🀹🀺🀻🀼🀽🀾🀿🁀🁁🁂🁃🁄🁅🁆🁇🁈🁉🁊🁋🁌🁍🁎🁏🁐🁑🁒🁓🁔🁕🁖🁗🁘🁙🁚🁛🁜🁝🁞🁟🁠🁡🁢🁣🁤🁥🁦🁧🁨🁩🁪🁫🁬🁭🁮🁯🁰🁱🁲🁳🁴🁵🁶🁷🁸🁹🁺🁻🁼🁽🁾🁿🂀🂁🂂🂃🂄🂅🂆🂇🂈🂉🂊🂋🂌🂍🂎🂏🂐🂑🂒🂓"s);
    allPassed &= test(L"😀😁😂😃😄😅😆😇😈😉😊😋😌😍😎😏😐😑😒😓😔😕😖😗😘😙😚😛😜😝😞😟😠😡😢😣😤😥😦😧😨😩😪😫😬😭😮😯😰😱😲😳😴😵😶😷😸😹😺😻😼😽😾😿🙀🙁🙂🙃🙄🙅🙆🙇🙈🙉🙊🙋🙌🙍🙎🙏"s, u8"😀😁😂😃😄😅😆😇😈😉😊😋😌😍😎😏😐😑😒😓😔😕😖😗😘😙😚😛😜😝😞😟😠😡😢😣😤😥😦😧😨😩😪😫😬😭😮😯😰😱😲😳😴😵😶😷😸😹😺😻😼😽😾😿🙀🙁🙂🙃🙄🙅🙆🙇🙈🙉🙊🙋🙌🙍🙎🙏"s);
    allPassed &= test(u8"😀😁😂😃😄😅😆😇😈😉😊😋😌😍😎😏😐😑😒😓😔😕😖😗😘😙😚😛😜😝😞😟😠😡😢😣😤😥😦😧😨😩😪😫😬😭😮😯😰😱😲😳😴😵😶😷😸😹😺😻😼😽😾😿🙀🙁🙂🙃🙄🙅🙆🙇🙈🙉🙊🙋🙌🙍🙎🙏"s, L"😀😁😂😃😄😅😆😇😈😉😊😋😌😍😎😏😐😑😒😓😔😕😖😗😘😙😚😛😜😝😞😟😠😡😢😣😤😥😦😧😨😩😪😫😬😭😮😯😰😱😲😳😴😵😶😷😸😹😺😻😼😽😾😿🙀🙁🙂🙃🙄🙅🙆🙇🙈🙉🙊🙋🙌🙍🙎🙏"s);
    allPassed &= test(L"𓀀𓀁𓀂𓀃𓀄𓀅𓀆𓀇𓀈𓀉𓀊𓀋𓀌𓀍𓀎𓀏𓀐𓀑𓀒𓀓𓀔𓀕𓀖𓀗𓀘𓀙𓀚𓀛𓀜𓀝𓀞𓀟𓀠𓀡𓀢𓀣𓀤𓀥𓀦𓀧𓀨𓀩𓀪𓀫𓀬𓀭𓀮𓀯𓀰𓀱𓀲𓀳𓀴𓀵𓀶𓀷𓀸𓀹𓀺𓀻𓀼𓀽𓀾𓀿𓁀𓁁𓁂𓁃𓁄𓁅𓁆𓁇𓁈𓁉𓁊𓁋𓁌𓁍𓁎𓁏𓁐𓁑𓁒𓁓𓁔𓁕𓁖𓁗𓁘𓁙𓁚𓁛𓁜𓁝𓁞𓁟𓁠𓁡𓁢𓁣𓁤𓁥𓁦𓁧𓁨𓁩𓁪𓁫𓁬𓁭𓁮𓁯𓁰𓁱𓁲𓁳𓁴𓁵𓁶𓁷𓁸𓁹𓁺𓁻𓁼𓁽𓁾𓁿𓂀𓂁𓂂𓂃𓂄𓂅𓂆𓂇𓂈𓂉𓂊𓂋𓂌𓂍𓂎𓂏𓂐𓂑𓂒𓂓𓂔𓂕𓂖𓂗𓂘𓂙𓂚𓂛𓂜𓂝𓂞𓂟𓂠𓂡𓂢𓂣𓂤𓂥𓂦𓂧𓂨𓂩𓂪𓂫𓂬𓂭𓂮𓂯𓂰𓂱𓂲𓂳𓂴𓂵𓂶𓂷𓂸𓂹𓂺𓂻𓂼𓂽𓂾𓂿𓃀𓃁𓃂𓃃𓃄𓃅𓃆𓃇𓃈𓃉𓃊𓃋𓃌𓃍𓃎𓃏𓃐𓃑𓃒𓃓𓃔𓃕𓃖𓃗𓃘𓃙𓃚𓃛𓃜𓃝𓃞𓃟𓃠𓃡𓃢𓃣𓃤𓃥𓃦𓃧𓃨𓃩𓃪𓃫𓃬𓃭𓃮𓃯𓃰𓃱𓃲𓃳𓃴𓃵𓃶𓃷𓃸𓃹𓃺𓃻𓃼𓃽𓃾𓃿𓄀"s, u8"𓀀𓀁𓀂𓀃𓀄𓀅𓀆𓀇𓀈𓀉𓀊𓀋𓀌𓀍𓀎𓀏𓀐𓀑𓀒𓀓𓀔𓀕𓀖𓀗𓀘𓀙𓀚𓀛𓀜𓀝𓀞𓀟𓀠𓀡𓀢𓀣𓀤𓀥𓀦𓀧𓀨𓀩𓀪𓀫𓀬𓀭𓀮𓀯𓀰𓀱𓀲𓀳𓀴𓀵𓀶𓀷𓀸𓀹𓀺𓀻𓀼𓀽𓀾𓀿𓁀𓁁𓁂𓁃𓁄𓁅𓁆𓁇𓁈𓁉𓁊𓁋𓁌𓁍𓁎𓁏𓁐𓁑𓁒𓁓𓁔𓁕𓁖𓁗𓁘𓁙𓁚𓁛𓁜𓁝𓁞𓁟𓁠𓁡𓁢𓁣𓁤𓁥𓁦𓁧𓁨𓁩𓁪𓁫𓁬𓁭𓁮𓁯𓁰𓁱𓁲𓁳𓁴𓁵𓁶𓁷𓁸𓁹𓁺𓁻𓁼𓁽𓁾𓁿𓂀𓂁𓂂𓂃𓂄𓂅𓂆𓂇𓂈𓂉𓂊𓂋𓂌𓂍𓂎𓂏𓂐𓂑𓂒𓂓𓂔𓂕𓂖𓂗𓂘𓂙𓂚𓂛𓂜𓂝𓂞𓂟𓂠𓂡𓂢𓂣𓂤𓂥𓂦𓂧𓂨𓂩𓂪𓂫𓂬𓂭𓂮𓂯𓂰𓂱𓂲𓂳𓂴𓂵𓂶𓂷𓂸𓂹𓂺𓂻𓂼𓂽𓂾𓂿𓃀𓃁𓃂𓃃𓃄𓃅𓃆𓃇𓃈𓃉𓃊𓃋𓃌𓃍𓃎𓃏𓃐𓃑𓃒𓃓𓃔𓃕𓃖𓃗𓃘𓃙𓃚𓃛𓃜𓃝𓃞𓃟𓃠𓃡𓃢𓃣𓃤𓃥𓃦𓃧𓃨𓃩𓃪𓃫𓃬𓃭𓃮𓃯𓃰𓃱𓃲𓃳𓃴𓃵𓃶𓃷𓃸𓃹𓃺𓃻𓃼𓃽𓃾𓃿𓄀"s);
    allPassed &= test(u8"𓀀𓀁𓀂𓀃𓀄𓀅𓀆𓀇𓀈𓀉𓀊𓀋𓀌𓀍𓀎𓀏𓀐𓀑𓀒𓀓𓀔𓀕𓀖𓀗𓀘𓀙𓀚𓀛𓀜𓀝𓀞𓀟𓀠𓀡𓀢𓀣𓀤𓀥𓀦𓀧𓀨𓀩𓀪𓀫𓀬𓀭𓀮𓀯𓀰𓀱𓀲𓀳𓀴𓀵𓀶𓀷𓀸𓀹𓀺𓀻𓀼𓀽𓀾𓀿𓁀𓁁𓁂𓁃𓁄𓁅𓁆𓁇𓁈𓁉𓁊𓁋𓁌𓁍𓁎𓁏𓁐𓁑𓁒𓁓𓁔𓁕𓁖𓁗𓁘𓁙𓁚𓁛𓁜𓁝𓁞𓁟𓁠𓁡𓁢𓁣𓁤𓁥𓁦𓁧𓁨𓁩𓁪𓁫𓁬𓁭𓁮𓁯𓁰𓁱𓁲𓁳𓁴𓁵𓁶𓁷𓁸𓁹𓁺𓁻𓁼𓁽𓁾𓁿𓂀𓂁𓂂𓂃𓂄𓂅𓂆𓂇𓂈𓂉𓂊𓂋𓂌𓂍𓂎𓂏𓂐𓂑𓂒𓂓𓂔𓂕𓂖𓂗𓂘𓂙𓂚𓂛𓂜𓂝𓂞𓂟𓂠𓂡𓂢𓂣𓂤𓂥𓂦𓂧𓂨𓂩𓂪𓂫𓂬𓂭𓂮𓂯𓂰𓂱𓂲𓂳𓂴𓂵𓂶𓂷𓂸𓂹𓂺𓂻𓂼𓂽𓂾𓂿𓃀𓃁𓃂𓃃𓃄𓃅𓃆𓃇𓃈𓃉𓃊𓃋𓃌𓃍𓃎𓃏𓃐𓃑𓃒𓃓𓃔𓃕𓃖𓃗𓃘𓃙𓃚𓃛𓃜𓃝𓃞𓃟𓃠𓃡𓃢𓃣𓃤𓃥𓃦𓃧𓃨𓃩𓃪𓃫𓃬𓃭𓃮𓃯𓃰𓃱𓃲𓃳𓃴𓃵𓃶𓃷𓃸𓃹𓃺𓃻𓃼𓃽𓃾𓃿𓄀"s, L"𓀀𓀁𓀂𓀃𓀄𓀅𓀆𓀇𓀈𓀉𓀊𓀋𓀌𓀍𓀎𓀏𓀐𓀑𓀒𓀓𓀔𓀕𓀖𓀗𓀘𓀙𓀚𓀛𓀜𓀝𓀞𓀟𓀠𓀡𓀢𓀣𓀤𓀥𓀦𓀧𓀨𓀩𓀪𓀫𓀬𓀭𓀮𓀯𓀰𓀱𓀲𓀳𓀴𓀵𓀶𓀷𓀸𓀹𓀺𓀻𓀼𓀽𓀾𓀿𓁀𓁁𓁂𓁃𓁄𓁅𓁆𓁇𓁈𓁉𓁊𓁋𓁌𓁍𓁎𓁏𓁐𓁑𓁒𓁓𓁔𓁕𓁖𓁗𓁘𓁙𓁚𓁛𓁜𓁝𓁞𓁟𓁠𓁡𓁢𓁣𓁤𓁥𓁦𓁧𓁨𓁩𓁪𓁫𓁬𓁭𓁮𓁯𓁰𓁱𓁲𓁳𓁴𓁵𓁶𓁷𓁸𓁹𓁺𓁻𓁼𓁽𓁾𓁿𓂀𓂁𓂂𓂃𓂄𓂅𓂆𓂇𓂈𓂉𓂊𓂋𓂌𓂍𓂎𓂏𓂐𓂑𓂒𓂓𓂔𓂕𓂖𓂗𓂘𓂙𓂚𓂛𓂜𓂝𓂞𓂟𓂠𓂡𓂢𓂣𓂤𓂥𓂦𓂧𓂨𓂩𓂪𓂫𓂬𓂭𓂮𓂯𓂰𓂱𓂲𓂳𓂴𓂵𓂶𓂷𓂸𓂹𓂺𓂻𓂼𓂽𓂾𓂿𓃀𓃁𓃂𓃃𓃄𓃅𓃆𓃇𓃈𓃉𓃊𓃋𓃌𓃍𓃎𓃏𓃐𓃑𓃒𓃓𓃔𓃕𓃖𓃗𓃘𓃙𓃚𓃛𓃜𓃝𓃞𓃟𓃠𓃡𓃢𓃣𓃤𓃥𓃦𓃧𓃨𓃩𓃪𓃫𓃬𓃭𓃮𓃯𓃰𓃱𓃲𓃳𓃴𓃵𓃶𓃷𓃸𓃹𓃺𓃻𓃼𓃽𓃾𓃿𓄀"s);

    // Supplementary Ideographic Plane
    allPassed &= test(L"𠀀𠀁𠀂𠀃𠀄𠀅𠀆𠀇𠀈𠀉𠀊𠀋𠀌𠀍𠀎𠀏𠀐𠀑𠀒𠀓𠀔𠀕𠀖𠀗𠀘𠀙𠀚𠀛𠀜𠀝𠀞𠀟𠀠𠀡𠀢𠀣𠀤𠀥𠀦𠀧𠀨𠀩𠀪𠀫𠀬𠀭𠀮𠀯𠀰𠀱𠀲𠀳𠀴𠀵𠀶𠀷𠀸𠀹𠀺𠀻𠀼𠀽𠀾𠀿𠁀𠁁𠁂𠁃𠁄𠁅𠁆𠁇𠁈𠁉𠁊𠁋𠁌𠁍𠁎𠁏𠁐𠁑𠁒𠁓𠁔𠁕𠁖𠁗𠁘𠁙𠁚𠁛𠁜𠁝𠁞𠁟𠁠𠁡𠁢𠁣𠁤𠁥𠁦𠁧𠁨𠁩𠁪𠁫𠁬𠁭𠁮𠁯𠁰𠁱𠁲𠁳𠁴𠁵𠁶𠁷𠁸𠁹𠁺𠁻𠁼𠁽𠁾𠁿𠂀𠂁𠂂𠂃𠂄𠂅𠂆𠂇𠂈𠂉𠂊𠂋𠂌𠂍𠂎𠂏𠂐𠂑𠂒𠂓𠂔𠂕𠂖𠂗𠂘𠂙𠂚𠂛𠂜𠂝𠂞𠂟𠂠𠂡𠂢𠂣𠂤𠂥𠂦𠂧𠂨𠂩𠂪𠂫𠂬𠂭𠂮𠂯𠂰𠂱𠂲𠂳𠂴𠂵𠂶𠂷𠂸𠂹𠂺𠂻𠂼𠂽𠂾𠂿𠃀𠃁𠃂𠃃𠃄𠃅𠃆𠃇𠃈𠃉𠃊𠃋𠃌𠃍𠃎𠃏𠃐𠃑𠃒𠃓𠃔𠃕𠃖𠃗𠃘𠃙𠃚𠃛𠃜𠃝𠃞𠃟𠃠𠃡𠃢𠃣𠃤𠃥𠃦𠃧𠃨𠃩𠃪𠃫𠃬𠃭𠃮𠃯𠃰𠃱𠃲𠃳𠃴𠃵𠃶𠃷𠃸𠃹𠃺𠃻𠃼𠃽𠃾𠃿𠄀"s, u8"𠀀𠀁𠀂𠀃𠀄𠀅𠀆𠀇𠀈𠀉𠀊𠀋𠀌𠀍𠀎𠀏𠀐𠀑𠀒𠀓𠀔𠀕𠀖𠀗𠀘𠀙𠀚𠀛𠀜𠀝𠀞𠀟𠀠𠀡𠀢𠀣𠀤𠀥𠀦𠀧𠀨𠀩𠀪𠀫𠀬𠀭𠀮𠀯𠀰𠀱𠀲𠀳𠀴𠀵𠀶𠀷𠀸𠀹𠀺𠀻𠀼𠀽𠀾𠀿𠁀𠁁𠁂𠁃𠁄𠁅𠁆𠁇𠁈𠁉𠁊𠁋𠁌𠁍𠁎𠁏𠁐𠁑𠁒𠁓𠁔𠁕𠁖𠁗𠁘𠁙𠁚𠁛𠁜𠁝𠁞𠁟𠁠𠁡𠁢𠁣𠁤𠁥𠁦𠁧𠁨𠁩𠁪𠁫𠁬𠁭𠁮𠁯𠁰𠁱𠁲𠁳𠁴𠁵𠁶𠁷𠁸𠁹𠁺𠁻𠁼𠁽𠁾𠁿𠂀𠂁𠂂𠂃𠂄𠂅𠂆𠂇𠂈𠂉𠂊𠂋𠂌𠂍𠂎𠂏𠂐𠂑𠂒𠂓𠂔𠂕𠂖𠂗𠂘𠂙𠂚𠂛𠂜𠂝𠂞𠂟𠂠𠂡𠂢𠂣𠂤𠂥𠂦𠂧𠂨𠂩𠂪𠂫𠂬𠂭𠂮𠂯𠂰𠂱𠂲𠂳𠂴𠂵𠂶𠂷𠂸𠂹𠂺𠂻𠂼𠂽𠂾𠂿𠃀𠃁𠃂𠃃𠃄𠃅𠃆𠃇𠃈𠃉𠃊𠃋𠃌𠃍𠃎𠃏𠃐𠃑𠃒𠃓𠃔𠃕𠃖𠃗𠃘𠃙𠃚𠃛𠃜𠃝𠃞𠃟𠃠𠃡𠃢𠃣𠃤𠃥𠃦𠃧𠃨𠃩𠃪𠃫𠃬𠃭𠃮𠃯𠃰𠃱𠃲𠃳𠃴𠃵𠃶𠃷𠃸𠃹𠃺𠃻𠃼𠃽𠃾𠃿𠄀"s);
    allPassed &= test(u8"𠀀𠀁𠀂𠀃𠀄𠀅𠀆𠀇𠀈𠀉𠀊𠀋𠀌𠀍𠀎𠀏𠀐𠀑𠀒𠀓𠀔𠀕𠀖𠀗𠀘𠀙𠀚𠀛𠀜𠀝𠀞𠀟𠀠𠀡𠀢𠀣𠀤𠀥𠀦𠀧𠀨𠀩𠀪𠀫𠀬𠀭𠀮𠀯𠀰𠀱𠀲𠀳𠀴𠀵𠀶𠀷𠀸𠀹𠀺𠀻𠀼𠀽𠀾𠀿𠁀𠁁𠁂𠁃𠁄𠁅𠁆𠁇𠁈𠁉𠁊𠁋𠁌𠁍𠁎𠁏𠁐𠁑𠁒𠁓𠁔𠁕𠁖𠁗𠁘𠁙𠁚𠁛𠁜𠁝𠁞𠁟𠁠𠁡𠁢𠁣𠁤𠁥𠁦𠁧𠁨𠁩𠁪𠁫𠁬𠁭𠁮𠁯𠁰𠁱𠁲𠁳𠁴𠁵𠁶𠁷𠁸𠁹𠁺𠁻𠁼𠁽𠁾𠁿𠂀𠂁𠂂𠂃𠂄𠂅𠂆𠂇𠂈𠂉𠂊𠂋𠂌𠂍𠂎𠂏𠂐𠂑𠂒𠂓𠂔𠂕𠂖𠂗𠂘𠂙𠂚𠂛𠂜𠂝𠂞𠂟𠂠𠂡𠂢𠂣𠂤𠂥𠂦𠂧𠂨𠂩𠂪𠂫𠂬𠂭𠂮𠂯𠂰𠂱𠂲𠂳𠂴𠂵𠂶𠂷𠂸𠂹𠂺𠂻𠂼𠂽𠂾𠂿𠃀𠃁𠃂𠃃𠃄𠃅𠃆𠃇𠃈𠃉𠃊𠃋𠃌𠃍𠃎𠃏𠃐𠃑𠃒𠃓𠃔𠃕𠃖𠃗𠃘𠃙𠃚𠃛𠃜𠃝𠃞𠃟𠃠𠃡𠃢𠃣𠃤𠃥𠃦𠃧𠃨𠃩𠃪𠃫𠃬𠃭𠃮𠃯𠃰𠃱𠃲𠃳𠃴𠃵𠃶𠃷𠃸𠃹𠃺𠃻𠃼𠃽𠃾𠃿𠄀"s, L"𠀀𠀁𠀂𠀃𠀄𠀅𠀆𠀇𠀈𠀉𠀊𠀋𠀌𠀍𠀎𠀏𠀐𠀑𠀒𠀓𠀔𠀕𠀖𠀗𠀘𠀙𠀚𠀛𠀜𠀝𠀞𠀟𠀠𠀡𠀢𠀣𠀤𠀥𠀦𠀧𠀨𠀩𠀪𠀫𠀬𠀭𠀮𠀯𠀰𠀱𠀲𠀳𠀴𠀵𠀶𠀷𠀸𠀹𠀺𠀻𠀼𠀽𠀾𠀿𠁀𠁁𠁂𠁃𠁄𠁅𠁆𠁇𠁈𠁉𠁊𠁋𠁌𠁍𠁎𠁏𠁐𠁑𠁒𠁓𠁔𠁕𠁖𠁗𠁘𠁙𠁚𠁛𠁜𠁝𠁞𠁟𠁠𠁡𠁢𠁣𠁤𠁥𠁦𠁧𠁨𠁩𠁪𠁫𠁬𠁭𠁮𠁯𠁰𠁱𠁲𠁳𠁴𠁵𠁶𠁷𠁸𠁹𠁺𠁻𠁼𠁽𠁾𠁿𠂀𠂁𠂂𠂃𠂄𠂅𠂆𠂇𠂈𠂉𠂊𠂋𠂌𠂍𠂎𠂏𠂐𠂑𠂒𠂓𠂔𠂕𠂖𠂗𠂘𠂙𠂚𠂛𠂜𠂝𠂞𠂟𠂠𠂡𠂢𠂣𠂤𠂥𠂦𠂧𠂨𠂩𠂪𠂫𠂬𠂭𠂮𠂯𠂰𠂱𠂲𠂳𠂴𠂵𠂶𠂷𠂸𠂹𠂺𠂻𠂼𠂽𠂾𠂿𠃀𠃁𠃂𠃃𠃄𠃅𠃆𠃇𠃈𠃉𠃊𠃋𠃌𠃍𠃎𠃏𠃐𠃑𠃒𠃓𠃔𠃕𠃖𠃗𠃘𠃙𠃚𠃛𠃜𠃝𠃞𠃟𠃠𠃡𠃢𠃣𠃤𠃥𠃦𠃧𠃨𠃩𠃪𠃫𠃬𠃭𠃮𠃯𠃰𠃱𠃲𠃳𠃴𠃵𠃶𠃷𠃸𠃹𠃺𠃻𠃼𠃽𠃾𠃿𠄀"s);
    allPassed &= test(L"𦘀𦘁𦘂𦘃𦘄𦘅𦘆𦘇𦘈𦘉𦘊𦘋𦘌𦘍𦘎𦘏𦘐𦘑𦘒𦘓𦘔𦘕𦘖𦘗𦘘𦘙𦘚𦘛𦘜𦘝𦘞𦘟𦘠𦘡𦘢𦘣𦘤𦘥𦘦𦘧𦘨𦘩𦘪𦘫𦘬𦘭𦘮𦘯𦘰𦘱𦘲𦘳𦘴𦘵𦘶𦘷𦘸𦘹𦘺𦘻𦘼𦘽𦘾𦘿𦙀𦙁𦙂𦙃𦙄𦙅𦙆𦙇𦙈𦙉𦙊𦙋𦙌𦙍𦙎𦙏𦙐𦙑𦙒𦙓𦙔𦙕𦙖𦙗𦙘𦙙𦙚𦙛𦙜𦙝𦙞𦙟𦙠𦙡𦙢𦙣𦙤𦙥𦙦𦙧𦙨𦙩𦙪𦙫𦙬𦙭𦙮𦙯𦙰𦙱𦙲𦙳𦙴𦙵𦙶𦙷𦙸𦙹𦙺𦙻𦙼𦙽𦙾𦙿𦚀𦚁𦚂𦚃𦚄𦚅𦚆𦚇𦚈𦚉𦚊𦚋𦚌𦚍𦚎𦚏𦚐𦚑𦚒𦚓𦚔𦚕𦚖𦚗𦚘𦚙𦚚𦚛𦚜𦚝𦚞𦚟𦚠𦚡𦚢𦚣𦚤𦚥𦚦𦚧𦚨𦚩𦚪𦚫𦚬𦚭𦚮𦚯𦚰𦚱𦚲𦚳𦚴𦚵𦚶𦚷𦚸𦚹𦚺𦚻𦚼𦚽𦚾𦚿𦛀𦛁𦛂𦛃𦛄𦛅𦛆𦛇𦛈𦛉𦛊𦛋𦛌𦛍𦛎𦛏𦛐𦛑𦛒𦛓𦛔𦛕𦛖𦛗𦛘𦛙𦛚𦛛𦛜𦛝𦛞𦛟𦛠𦛡𦛢𦛣𦛤𦛥𦛦𦛧𦛨𦛩𦛪𦛫𦛬𦛭𦛮𦛯𦛰𦛱𦛲𦛳𦛴𦛵𦛶𦛷𦛸𦛹𦛺𦛻𦛼𦛽𦛾𦛿𦜀"s, u8"𦘀𦘁𦘂𦘃𦘄𦘅𦘆𦘇𦘈𦘉𦘊𦘋𦘌𦘍𦘎𦘏𦘐𦘑𦘒𦘓𦘔𦘕𦘖𦘗𦘘𦘙𦘚𦘛𦘜𦘝𦘞𦘟𦘠𦘡𦘢𦘣𦘤𦘥𦘦𦘧𦘨𦘩𦘪𦘫𦘬𦘭𦘮𦘯𦘰𦘱𦘲𦘳𦘴𦘵𦘶𦘷𦘸𦘹𦘺𦘻𦘼𦘽𦘾𦘿𦙀𦙁𦙂𦙃𦙄𦙅𦙆𦙇𦙈𦙉𦙊𦙋𦙌𦙍𦙎𦙏𦙐𦙑𦙒𦙓𦙔𦙕𦙖𦙗𦙘𦙙𦙚𦙛𦙜𦙝𦙞𦙟𦙠𦙡𦙢𦙣𦙤𦙥𦙦𦙧𦙨𦙩𦙪𦙫𦙬𦙭𦙮𦙯𦙰𦙱𦙲𦙳𦙴𦙵𦙶𦙷𦙸𦙹𦙺𦙻𦙼𦙽𦙾𦙿𦚀𦚁𦚂𦚃𦚄𦚅𦚆𦚇𦚈𦚉𦚊𦚋𦚌𦚍𦚎𦚏𦚐𦚑𦚒𦚓𦚔𦚕𦚖𦚗𦚘𦚙𦚚𦚛𦚜𦚝𦚞𦚟𦚠𦚡𦚢𦚣𦚤𦚥𦚦𦚧𦚨𦚩𦚪𦚫𦚬𦚭𦚮𦚯𦚰𦚱𦚲𦚳𦚴𦚵𦚶𦚷𦚸𦚹𦚺𦚻𦚼𦚽𦚾𦚿𦛀𦛁𦛂𦛃𦛄𦛅𦛆𦛇𦛈𦛉𦛊𦛋𦛌𦛍𦛎𦛏𦛐𦛑𦛒𦛓𦛔𦛕𦛖𦛗𦛘𦛙𦛚𦛛𦛜𦛝𦛞𦛟𦛠𦛡𦛢𦛣𦛤𦛥𦛦𦛧𦛨𦛩𦛪𦛫𦛬𦛭𦛮𦛯𦛰𦛱𦛲𦛳𦛴𦛵𦛶𦛷𦛸𦛹𦛺𦛻𦛼𦛽𦛾𦛿𦜀"s);
    allPassed &= test(u8"𦘀𦘁𦘂𦘃𦘄𦘅𦘆𦘇𦘈𦘉𦘊𦘋𦘌𦘍𦘎𦘏𦘐𦘑𦘒𦘓𦘔𦘕𦘖𦘗𦘘𦘙𦘚𦘛𦘜𦘝𦘞𦘟𦘠𦘡𦘢𦘣𦘤𦘥𦘦𦘧𦘨𦘩𦘪𦘫𦘬𦘭𦘮𦘯𦘰𦘱𦘲𦘳𦘴𦘵𦘶𦘷𦘸𦘹𦘺𦘻𦘼𦘽𦘾𦘿𦙀𦙁𦙂𦙃𦙄𦙅𦙆𦙇𦙈𦙉𦙊𦙋𦙌𦙍𦙎𦙏𦙐𦙑𦙒𦙓𦙔𦙕𦙖𦙗𦙘𦙙𦙚𦙛𦙜𦙝𦙞𦙟𦙠𦙡𦙢𦙣𦙤𦙥𦙦𦙧𦙨𦙩𦙪𦙫𦙬𦙭𦙮𦙯𦙰𦙱𦙲𦙳𦙴𦙵𦙶𦙷𦙸𦙹𦙺𦙻𦙼𦙽𦙾𦙿𦚀𦚁𦚂𦚃𦚄𦚅𦚆𦚇𦚈𦚉𦚊𦚋𦚌𦚍𦚎𦚏𦚐𦚑𦚒𦚓𦚔𦚕𦚖𦚗𦚘𦚙𦚚𦚛𦚜𦚝𦚞𦚟𦚠𦚡𦚢𦚣𦚤𦚥𦚦𦚧𦚨𦚩𦚪𦚫𦚬𦚭𦚮𦚯𦚰𦚱𦚲𦚳𦚴𦚵𦚶𦚷𦚸𦚹𦚺𦚻𦚼𦚽𦚾𦚿𦛀𦛁𦛂𦛃𦛄𦛅𦛆𦛇𦛈𦛉𦛊𦛋𦛌𦛍𦛎𦛏𦛐𦛑𦛒𦛓𦛔𦛕𦛖𦛗𦛘𦛙𦛚𦛛𦛜𦛝𦛞𦛟𦛠𦛡𦛢𦛣𦛤𦛥𦛦𦛧𦛨𦛩𦛪𦛫𦛬𦛭𦛮𦛯𦛰𦛱𦛲𦛳𦛴𦛵𦛶𦛷𦛸𦛹𦛺𦛻𦛼𦛽𦛾𦛿𦜀"s, L"𦘀𦘁𦘂𦘃𦘄𦘅𦘆𦘇𦘈𦘉𦘊𦘋𦘌𦘍𦘎𦘏𦘐𦘑𦘒𦘓𦘔𦘕𦘖𦘗𦘘𦘙𦘚𦘛𦘜𦘝𦘞𦘟𦘠𦘡𦘢𦘣𦘤𦘥𦘦𦘧𦘨𦘩𦘪𦘫𦘬𦘭𦘮𦘯𦘰𦘱𦘲𦘳𦘴𦘵𦘶𦘷𦘸𦘹𦘺𦘻𦘼𦘽𦘾𦘿𦙀𦙁𦙂𦙃𦙄𦙅𦙆𦙇𦙈𦙉𦙊𦙋𦙌𦙍𦙎𦙏𦙐𦙑𦙒𦙓𦙔𦙕𦙖𦙗𦙘𦙙𦙚𦙛𦙜𦙝𦙞𦙟𦙠𦙡𦙢𦙣𦙤𦙥𦙦𦙧𦙨𦙩𦙪𦙫𦙬𦙭𦙮𦙯𦙰𦙱𦙲𦙳𦙴𦙵𦙶𦙷𦙸𦙹𦙺𦙻𦙼𦙽𦙾𦙿𦚀𦚁𦚂𦚃𦚄𦚅𦚆𦚇𦚈𦚉𦚊𦚋𦚌𦚍𦚎𦚏𦚐𦚑𦚒𦚓𦚔𦚕𦚖𦚗𦚘𦚙𦚚𦚛𦚜𦚝𦚞𦚟𦚠𦚡𦚢𦚣𦚤𦚥𦚦𦚧𦚨𦚩𦚪𦚫𦚬𦚭𦚮𦚯𦚰𦚱𦚲𦚳𦚴𦚵𦚶𦚷𦚸𦚹𦚺𦚻𦚼𦚽𦚾𦚿𦛀𦛁𦛂𦛃𦛄𦛅𦛆𦛇𦛈𦛉𦛊𦛋𦛌𦛍𦛎𦛏𦛐𦛑𦛒𦛓𦛔𦛕𦛖𦛗𦛘𦛙𦛚𦛛𦛜𦛝𦛞𦛟𦛠𦛡𦛢𦛣𦛤𦛥𦛦𦛧𦛨𦛩𦛪𦛫𦛬𦛭𦛮𦛯𦛰𦛱𦛲𦛳𦛴𦛵𦛶𦛷𦛸𦛹𦛺𦛻𦛼𦛽𦛾𦛿𦜀"s);
    allPassed &= test(L"𪤀𪤁𪤂𪤃𪤄𪤅𪤆𪤇𪤈𪤉𪤊𪤋𪤌𪤍𪤎𪤏𪤐𪤑𪤒𪤓𪤔𪤕𪤖𪤗𪤘𪤙𪤚𪤛𪤜𪤝𪤞𪤟𪤠𪤡𪤢𪤣𪤤𪤥𪤦𪤧𪤨𪤩𪤪𪤫𪤬𪤭𪤮𪤯𪤰𪤱𪤲𪤳𪤴𪤵𪤶𪤷𪤸𪤹𪤺𪤻𪤼𪤽𪤾𪤿𪥀𪥁𪥂𪥃𪥄𪥅𪥆𪥇𪥈𪥉𪥊𪥋𪥌𪥍𪥎𪥏𪥐𪥑𪥒𪥓𪥔𪥕𪥖𪥗𪥘𪥙𪥚𪥛𪥜𪥝𪥞𪥟𪥠𪥡𪥢𪥣𪥤𪥥𪥦𪥧𪥨𪥩𪥪𪥫𪥬𪥭𪥮𪥯𪥰𪥱𪥲𪥳𪥴𪥵𪥶𪥷𪥸𪥹𪥺𪥻𪥼𪥽𪥾𪥿𪦀𪦁𪦂𪦃𪦄𪦅𪦆𪦇𪦈𪦉𪦊𪦋𪦌𪦍𪦎𪦏𪦐𪦑𪦒𪦓𪦔𪦕𪦖𪦗𪦘𪦙𪦚𪦛𪦜𪦝𪦞𪦟𪦠𪦡𪦢𪦣𪦤𪦥𪦦𪦧𪦨𪦩𪦪𪦫𪦬𪦭𪦮𪦯𪦰𪦱𪦲𪦳𪦴𪦵𪦶𪦷𪦸𪦹𪦺𪦻𪦼𪦽𪦾𪦿𪧀𪧁𪧂𪧃𪧄𪧅𪧆𪧇𪧈𪧉𪧊𪧋𪧌𪧍𪧎𪧏𪧐𪧑𪧒𪧓𪧔𪧕𪧖𪧗𪧘𪧙𪧚𪧛𪧜𪧝𪧞𪧟𪧠𪧡𪧢𪧣𪧤𪧥𪧦𪧧𪧨𪧩𪧪𪧫𪧬𪧭𪧮𪧯𪧰𪧱𪧲𪧳𪧴𪧵𪧶𪧷𪧸𪧹𪧺𪧻𪧼𪧽𪧾𪧿𪨀"s, u8"𪤀𪤁𪤂𪤃𪤄𪤅𪤆𪤇𪤈𪤉𪤊𪤋𪤌𪤍𪤎𪤏𪤐𪤑𪤒𪤓𪤔𪤕𪤖𪤗𪤘𪤙𪤚𪤛𪤜𪤝𪤞𪤟𪤠𪤡𪤢𪤣𪤤𪤥𪤦𪤧𪤨𪤩𪤪𪤫𪤬𪤭𪤮𪤯𪤰𪤱𪤲𪤳𪤴𪤵𪤶𪤷𪤸𪤹𪤺𪤻𪤼𪤽𪤾𪤿𪥀𪥁𪥂𪥃𪥄𪥅𪥆𪥇𪥈𪥉𪥊𪥋𪥌𪥍𪥎𪥏𪥐𪥑𪥒𪥓𪥔𪥕𪥖𪥗𪥘𪥙𪥚𪥛𪥜𪥝𪥞𪥟𪥠𪥡𪥢𪥣𪥤𪥥𪥦𪥧𪥨𪥩𪥪𪥫𪥬𪥭𪥮𪥯𪥰𪥱𪥲𪥳𪥴𪥵𪥶𪥷𪥸𪥹𪥺𪥻𪥼𪥽𪥾𪥿𪦀𪦁𪦂𪦃𪦄𪦅𪦆𪦇𪦈𪦉𪦊𪦋𪦌𪦍𪦎𪦏𪦐𪦑𪦒𪦓𪦔𪦕𪦖𪦗𪦘𪦙𪦚𪦛𪦜𪦝𪦞𪦟𪦠𪦡𪦢𪦣𪦤𪦥𪦦𪦧𪦨𪦩𪦪𪦫𪦬𪦭𪦮𪦯𪦰𪦱𪦲𪦳𪦴𪦵𪦶𪦷𪦸𪦹𪦺𪦻𪦼𪦽𪦾𪦿𪧀𪧁𪧂𪧃𪧄𪧅𪧆𪧇𪧈𪧉𪧊𪧋𪧌𪧍𪧎𪧏𪧐𪧑𪧒𪧓𪧔𪧕𪧖𪧗𪧘𪧙𪧚𪧛𪧜𪧝𪧞𪧟𪧠𪧡𪧢𪧣𪧤𪧥𪧦𪧧𪧨𪧩𪧪𪧫𪧬𪧭𪧮𪧯𪧰𪧱𪧲𪧳𪧴𪧵𪧶𪧷𪧸𪧹𪧺𪧻𪧼𪧽𪧾𪧿𪨀"s);
    allPassed &= test(u8"𪤀𪤁𪤂𪤃𪤄𪤅𪤆𪤇𪤈𪤉𪤊𪤋𪤌𪤍𪤎𪤏𪤐𪤑𪤒𪤓𪤔𪤕𪤖𪤗𪤘𪤙𪤚𪤛𪤜𪤝𪤞𪤟𪤠𪤡𪤢𪤣𪤤𪤥𪤦𪤧𪤨𪤩𪤪𪤫𪤬𪤭𪤮𪤯𪤰𪤱𪤲𪤳𪤴𪤵𪤶𪤷𪤸𪤹𪤺𪤻𪤼𪤽𪤾𪤿𪥀𪥁𪥂𪥃𪥄𪥅𪥆𪥇𪥈𪥉𪥊𪥋𪥌𪥍𪥎𪥏𪥐𪥑𪥒𪥓𪥔𪥕𪥖𪥗𪥘𪥙𪥚𪥛𪥜𪥝𪥞𪥟𪥠𪥡𪥢𪥣𪥤𪥥𪥦𪥧𪥨𪥩𪥪𪥫𪥬𪥭𪥮𪥯𪥰𪥱𪥲𪥳𪥴𪥵𪥶𪥷𪥸𪥹𪥺𪥻𪥼𪥽𪥾𪥿𪦀𪦁𪦂𪦃𪦄𪦅𪦆𪦇𪦈𪦉𪦊𪦋𪦌𪦍𪦎𪦏𪦐𪦑𪦒𪦓𪦔𪦕𪦖𪦗𪦘𪦙𪦚𪦛𪦜𪦝𪦞𪦟𪦠𪦡𪦢𪦣𪦤𪦥𪦦𪦧𪦨𪦩𪦪𪦫𪦬𪦭𪦮𪦯𪦰𪦱𪦲𪦳𪦴𪦵𪦶𪦷𪦸𪦹𪦺𪦻𪦼𪦽𪦾𪦿𪧀𪧁𪧂𪧃𪧄𪧅𪧆𪧇𪧈𪧉𪧊𪧋𪧌𪧍𪧎𪧏𪧐𪧑𪧒𪧓𪧔𪧕𪧖𪧗𪧘𪧙𪧚𪧛𪧜𪧝𪧞𪧟𪧠𪧡𪧢𪧣𪧤𪧥𪧦𪧧𪧨𪧩𪧪𪧫𪧬𪧭𪧮𪧯𪧰𪧱𪧲𪧳𪧴𪧵𪧶𪧷𪧸𪧹𪧺𪧻𪧼𪧽𪧾𪧿𪨀"s, L"𪤀𪤁𪤂𪤃𪤄𪤅𪤆𪤇𪤈𪤉𪤊𪤋𪤌𪤍𪤎𪤏𪤐𪤑𪤒𪤓𪤔𪤕𪤖𪤗𪤘𪤙𪤚𪤛𪤜𪤝𪤞𪤟𪤠𪤡𪤢𪤣𪤤𪤥𪤦𪤧𪤨𪤩𪤪𪤫𪤬𪤭𪤮𪤯𪤰𪤱𪤲𪤳𪤴𪤵𪤶𪤷𪤸𪤹𪤺𪤻𪤼𪤽𪤾𪤿𪥀𪥁𪥂𪥃𪥄𪥅𪥆𪥇𪥈𪥉𪥊𪥋𪥌𪥍𪥎𪥏𪥐𪥑𪥒𪥓𪥔𪥕𪥖𪥗𪥘𪥙𪥚𪥛𪥜𪥝𪥞𪥟𪥠𪥡𪥢𪥣𪥤𪥥𪥦𪥧𪥨𪥩𪥪𪥫𪥬𪥭𪥮𪥯𪥰𪥱𪥲𪥳𪥴𪥵𪥶𪥷𪥸𪥹𪥺𪥻𪥼𪥽𪥾𪥿𪦀𪦁𪦂𪦃𪦄𪦅𪦆𪦇𪦈𪦉𪦊𪦋𪦌𪦍𪦎𪦏𪦐𪦑𪦒𪦓𪦔𪦕𪦖𪦗𪦘𪦙𪦚𪦛𪦜𪦝𪦞𪦟𪦠𪦡𪦢𪦣𪦤𪦥𪦦𪦧𪦨𪦩𪦪𪦫𪦬𪦭𪦮𪦯𪦰𪦱𪦲𪦳𪦴𪦵𪦶𪦷𪦸𪦹𪦺𪦻𪦼𪦽𪦾𪦿𪧀𪧁𪧂𪧃𪧄𪧅𪧆𪧇𪧈𪧉𪧊𪧋𪧌𪧍𪧎𪧏𪧐𪧑𪧒𪧓𪧔𪧕𪧖𪧗𪧘𪧙𪧚𪧛𪧜𪧝𪧞𪧟𪧠𪧡𪧢𪧣𪧤𪧥𪧦𪧧𪧨𪧩𪧪𪧫𪧬𪧭𪧮𪧯𪧰𪧱𪧲𪧳𪧴𪧵𪧶𪧷𪧸𪧹𪧺𪧻𪧼𪧽𪧾𪧿𪨀"s);
    allPassed &= test(L"𫝀𫝁𫝂𫝃𫝄𫝅𫝆𫝇𫝈𫝉𫝊𫝋𫝌𫝍𫝎𫝏𫝐𫝑𫝒𫝓𫝔𫝕𫝖𫝗𫝘𫝙𫝚𫝛𫝜𫝝𫝞𫝟𫝠𫝡𫝢𫝣𫝤𫝥𫝦𫝧𫝨𫝩𫝪𫝫𫝬𫝭𫝮𫝯𫝰𫝱𫝲𫝳𫝴𫝵𫝶𫝷𫝸𫝹𫝺𫝻𫝼𫝽𫝾𫝿𫞀𫞁𫞂𫞃𫞄𫞅𫞆𫞇𫞈𫞉𫞊𫞋𫞌𫞍𫞎𫞏𫞐𫞑𫞒𫞓𫞔𫞕𫞖𫞗𫞘𫞙𫞚𫞛𫞜𫞝𫞞𫞟𫞠𫞡𫞢𫞣𫞤𫞥𫞦𫞧𫞨𫞩𫞪𫞫𫞬𫞭𫞮𫞯𫞰𫞱𫞲𫞳𫞴𫞵𫞶𫞷𫞸𫞹𫞺𫞻𫞼𫞽𫞾𫞿𫟀𫟁𫟂𫟃𫟄𫟅𫟆𫟇𫟈𫟉𫟊𫟋𫟌𫟍𫟎𫟏𫟐𫟑𫟒𫟓𫟔𫟕𫟖𫟗𫟘𫟙𫟚𫟛𫟜𫟝𫟞𫟟𫟠𫟡𫟢𫟣𫟤𫟥𫟦𫟧𫟨𫟩𫟪𫟫𫟬𫟭𫟮𫟯𫟰𫟱𫟲𫟳𫟴𫟵𫟶𫟷𫟸𫟹𫟺𫟻𫟼𫟽𫟾𫟿𫠀𫠁𫠂𫠃𫠄𫠅𫠆𫠇𫠈𫠉𫠊𫠋𫠌𫠍𫠎𫠏𫠐𫠑𫠒𫠓𫠔𫠕𫠖𫠗𫠘𫠙𫠚𫠛𫠜𫠝"s, u8"𫝀𫝁𫝂𫝃𫝄𫝅𫝆𫝇𫝈𫝉𫝊𫝋𫝌𫝍𫝎𫝏𫝐𫝑𫝒𫝓𫝔𫝕𫝖𫝗𫝘𫝙𫝚𫝛𫝜𫝝𫝞𫝟𫝠𫝡𫝢𫝣𫝤𫝥𫝦𫝧𫝨𫝩𫝪𫝫𫝬𫝭𫝮𫝯𫝰𫝱𫝲𫝳𫝴𫝵𫝶𫝷𫝸𫝹𫝺𫝻𫝼𫝽𫝾𫝿𫞀𫞁𫞂𫞃𫞄𫞅𫞆𫞇𫞈𫞉𫞊𫞋𫞌𫞍𫞎𫞏𫞐𫞑𫞒𫞓𫞔𫞕𫞖𫞗𫞘𫞙𫞚𫞛𫞜𫞝𫞞𫞟𫞠𫞡𫞢𫞣𫞤𫞥𫞦𫞧𫞨𫞩𫞪𫞫𫞬𫞭𫞮𫞯𫞰𫞱𫞲𫞳𫞴𫞵𫞶𫞷𫞸𫞹𫞺𫞻𫞼𫞽𫞾𫞿𫟀𫟁𫟂𫟃𫟄𫟅𫟆𫟇𫟈𫟉𫟊𫟋𫟌𫟍𫟎𫟏𫟐𫟑𫟒𫟓𫟔𫟕𫟖𫟗𫟘𫟙𫟚𫟛𫟜𫟝𫟞𫟟𫟠𫟡𫟢𫟣𫟤𫟥𫟦𫟧𫟨𫟩𫟪𫟫𫟬𫟭𫟮𫟯𫟰𫟱𫟲𫟳𫟴𫟵𫟶𫟷𫟸𫟹𫟺𫟻𫟼𫟽𫟾𫟿𫠀𫠁𫠂𫠃𫠄𫠅𫠆𫠇𫠈𫠉𫠊𫠋𫠌𫠍𫠎𫠏𫠐𫠑𫠒𫠓𫠔𫠕𫠖𫠗𫠘𫠙𫠚𫠛𫠜𫠝"s);
    allPassed &= test(u8"𫝀𫝁𫝂𫝃𫝄𫝅𫝆𫝇𫝈𫝉𫝊𫝋𫝌𫝍𫝎𫝏𫝐𫝑𫝒𫝓𫝔𫝕𫝖𫝗𫝘𫝙𫝚𫝛𫝜𫝝𫝞𫝟𫝠𫝡𫝢𫝣𫝤𫝥𫝦𫝧𫝨𫝩𫝪𫝫𫝬𫝭𫝮𫝯𫝰𫝱𫝲𫝳𫝴𫝵𫝶𫝷𫝸𫝹𫝺𫝻𫝼𫝽𫝾𫝿𫞀𫞁𫞂𫞃𫞄𫞅𫞆𫞇𫞈𫞉𫞊𫞋𫞌𫞍𫞎𫞏𫞐𫞑𫞒𫞓𫞔𫞕𫞖𫞗𫞘𫞙𫞚𫞛𫞜𫞝𫞞𫞟𫞠𫞡𫞢𫞣𫞤𫞥𫞦𫞧𫞨𫞩𫞪𫞫𫞬𫞭𫞮𫞯𫞰𫞱𫞲𫞳𫞴𫞵𫞶𫞷𫞸𫞹𫞺𫞻𫞼𫞽𫞾𫞿𫟀𫟁𫟂𫟃𫟄𫟅𫟆𫟇𫟈𫟉𫟊𫟋𫟌𫟍𫟎𫟏𫟐𫟑𫟒𫟓𫟔𫟕𫟖𫟗𫟘𫟙𫟚𫟛𫟜𫟝𫟞𫟟𫟠𫟡𫟢𫟣𫟤𫟥𫟦𫟧𫟨𫟩𫟪𫟫𫟬𫟭𫟮𫟯𫟰𫟱𫟲𫟳𫟴𫟵𫟶𫟷𫟸𫟹𫟺𫟻𫟼𫟽𫟾𫟿𫠀𫠁𫠂𫠃𫠄𫠅𫠆𫠇𫠈𫠉𫠊𫠋𫠌𫠍𫠎𫠏𫠐𫠑𫠒𫠓𫠔𫠕𫠖𫠗𫠘𫠙𫠚𫠛𫠜𫠝"s, L"𫝀𫝁𫝂𫝃𫝄𫝅𫝆𫝇𫝈𫝉𫝊𫝋𫝌𫝍𫝎𫝏𫝐𫝑𫝒𫝓𫝔𫝕𫝖𫝗𫝘𫝙𫝚𫝛𫝜𫝝𫝞𫝟𫝠𫝡𫝢𫝣𫝤𫝥𫝦𫝧𫝨𫝩𫝪𫝫𫝬𫝭𫝮𫝯𫝰𫝱𫝲𫝳𫝴𫝵𫝶𫝷𫝸𫝹𫝺𫝻𫝼𫝽𫝾𫝿𫞀𫞁𫞂𫞃𫞄𫞅𫞆𫞇𫞈𫞉𫞊𫞋𫞌𫞍𫞎𫞏𫞐𫞑𫞒𫞓𫞔𫞕𫞖𫞗𫞘𫞙𫞚𫞛𫞜𫞝𫞞𫞟𫞠𫞡𫞢𫞣𫞤𫞥𫞦𫞧𫞨𫞩𫞪𫞫𫞬𫞭𫞮𫞯𫞰𫞱𫞲𫞳𫞴𫞵𫞶𫞷𫞸𫞹𫞺𫞻𫞼𫞽𫞾𫞿𫟀𫟁𫟂𫟃𫟄𫟅𫟆𫟇𫟈𫟉𫟊𫟋𫟌𫟍𫟎𫟏𫟐𫟑𫟒𫟓𫟔𫟕𫟖𫟗𫟘𫟙𫟚𫟛𫟜𫟝𫟞𫟟𫟠𫟡𫟢𫟣𫟤𫟥𫟦𫟧𫟨𫟩𫟪𫟫𫟬𫟭𫟮𫟯𫟰𫟱𫟲𫟳𫟴𫟵𫟶𫟷𫟸𫟹𫟺𫟻𫟼𫟽𫟾𫟿𫠀𫠁𫠂𫠃𫠄𫠅𫠆𫠇𫠈𫠉𫠊𫠋𫠌𫠍𫠎𫠏𫠐𫠑𫠒𫠓𫠔𫠕𫠖𫠗𫠘𫠙𫠚𫠛𫠜𫠝"s);

    // An unassigned plane with some more CJK characters in it
    allPassed &= test(L"𰀀𰀁𰀂𰀃𰀄𰀅𰀆𰀇𰀈𰀉𰀊𰀋𰀌𰀍𰀎𰀏𰀐𰀑𰀒𰀓𰀔𰀕𰀖𰀗𰀘𰀙𰀚𰀛𰀜𰀝𰀞𰀟𰀠𰀡𰀢𰀣𰀤𰀥𰀦𰀧𰀨𰀩𰀪𰀫𰀬𰀭𰀮𰀯𰀰𰀱𰀲𰀳𰀴𰀵𰀶𰀷𰀸𰀹𰀺𰀻𰀼𰀽𰀾𰀿𰁀𰁁𰁂𰁃𰁄𰁅𰁆𰁇𰁈𰁉𰁊𰁋𰁌𰁍𰁎𰁏𰁐𰁑𰁒𰁓𰁔𰁕𰁖𰁗𰁘𰁙𰁚𰁛𰁜𰁝𰁞𰁟𰁠𰁡𰁢𰁣𰁤𰁥𰁦𰁧𰁨𰁩𰁪𰁫𰁬𰁭𰁮𰁯𰁰𰁱𰁲𰁳𰁴𰁵𰁶𰁷𰁸𰁹𰁺𰁻𰁼𰁽𰁾𰁿𰂀𰂁𰂂𰂃𰂄𰂅𰂆𰂇𰂈𰂉𰂊𰂋𰂌𰂍𰂎𰂏𰂐𰂑𰂒𰂓𰂔𰂕𰂖𰂗𰂘𰂙𰂚𰂛𰂜𰂝𰂞𰂟𰂠𰂡𰂢𰂣𰂤𰂥𰂦𰂧𰂨𰂩𰂪𰂫𰂬𰂭𰂮𰂯𰂰𰂱𰂲𰂳𰂴𰂵𰂶𰂷𰂸𰂹𰂺𰂻𰂼𰂽𰂾𰂿𰃀𰃁𰃂𰃃𰃄𰃅𰃆𰃇𰃈𰃉𰃊𰃋𰃌𰃍𰃎𰃏𰃐𰃑𰃒𰃓𰃔𰃕𰃖𰃗𰃘𰃙𰃚𰃛𰃜𰃝𰃞𰃟𰃠𰃡𰃢𰃣𰃤𰃥𰃦𰃧𰃨𰃩𰃪𰃫𰃬𰃭𰃮𰃯𰃰𰃱𰃲𰃳𰃴𰃵𰃶𰃷𰃸𰃹𰃺𰃻𰃼𰃽𰃾𰃿𰄀"s, u8"𰀀𰀁𰀂𰀃𰀄𰀅𰀆𰀇𰀈𰀉𰀊𰀋𰀌𰀍𰀎𰀏𰀐𰀑𰀒𰀓𰀔𰀕𰀖𰀗𰀘𰀙𰀚𰀛𰀜𰀝𰀞𰀟𰀠𰀡𰀢𰀣𰀤𰀥𰀦𰀧𰀨𰀩𰀪𰀫𰀬𰀭𰀮𰀯𰀰𰀱𰀲𰀳𰀴𰀵𰀶𰀷𰀸𰀹𰀺𰀻𰀼𰀽𰀾𰀿𰁀𰁁𰁂𰁃𰁄𰁅𰁆𰁇𰁈𰁉𰁊𰁋𰁌𰁍𰁎𰁏𰁐𰁑𰁒𰁓𰁔𰁕𰁖𰁗𰁘𰁙𰁚𰁛𰁜𰁝𰁞𰁟𰁠𰁡𰁢𰁣𰁤𰁥𰁦𰁧𰁨𰁩𰁪𰁫𰁬𰁭𰁮𰁯𰁰𰁱𰁲𰁳𰁴𰁵𰁶𰁷𰁸𰁹𰁺𰁻𰁼𰁽𰁾𰁿𰂀𰂁𰂂𰂃𰂄𰂅𰂆𰂇𰂈𰂉𰂊𰂋𰂌𰂍𰂎𰂏𰂐𰂑𰂒𰂓𰂔𰂕𰂖𰂗𰂘𰂙𰂚𰂛𰂜𰂝𰂞𰂟𰂠𰂡𰂢𰂣𰂤𰂥𰂦𰂧𰂨𰂩𰂪𰂫𰂬𰂭𰂮𰂯𰂰𰂱𰂲𰂳𰂴𰂵𰂶𰂷𰂸𰂹𰂺𰂻𰂼𰂽𰂾𰂿𰃀𰃁𰃂𰃃𰃄𰃅𰃆𰃇𰃈𰃉𰃊𰃋𰃌𰃍𰃎𰃏𰃐𰃑𰃒𰃓𰃔𰃕𰃖𰃗𰃘𰃙𰃚𰃛𰃜𰃝𰃞𰃟𰃠𰃡𰃢𰃣𰃤𰃥𰃦𰃧𰃨𰃩𰃪𰃫𰃬𰃭𰃮𰃯𰃰𰃱𰃲𰃳𰃴𰃵𰃶𰃷𰃸𰃹𰃺𰃻𰃼𰃽𰃾𰃿𰄀"s);
    allPassed &= test(u8"𰀀𰀁𰀂𰀃𰀄𰀅𰀆𰀇𰀈𰀉𰀊𰀋𰀌𰀍𰀎𰀏𰀐𰀑𰀒𰀓𰀔𰀕𰀖𰀗𰀘𰀙𰀚𰀛𰀜𰀝𰀞𰀟𰀠𰀡𰀢𰀣𰀤𰀥𰀦𰀧𰀨𰀩𰀪𰀫𰀬𰀭𰀮𰀯𰀰𰀱𰀲𰀳𰀴𰀵𰀶𰀷𰀸𰀹𰀺𰀻𰀼𰀽𰀾𰀿𰁀𰁁𰁂𰁃𰁄𰁅𰁆𰁇𰁈𰁉𰁊𰁋𰁌𰁍𰁎𰁏𰁐𰁑𰁒𰁓𰁔𰁕𰁖𰁗𰁘𰁙𰁚𰁛𰁜𰁝𰁞𰁟𰁠𰁡𰁢𰁣𰁤𰁥𰁦𰁧𰁨𰁩𰁪𰁫𰁬𰁭𰁮𰁯𰁰𰁱𰁲𰁳𰁴𰁵𰁶𰁷𰁸𰁹𰁺𰁻𰁼𰁽𰁾𰁿𰂀𰂁𰂂𰂃𰂄𰂅𰂆𰂇𰂈𰂉𰂊𰂋𰂌𰂍𰂎𰂏𰂐𰂑𰂒𰂓𰂔𰂕𰂖𰂗𰂘𰂙𰂚𰂛𰂜𰂝𰂞𰂟𰂠𰂡𰂢𰂣𰂤𰂥𰂦𰂧𰂨𰂩𰂪𰂫𰂬𰂭𰂮𰂯𰂰𰂱𰂲𰂳𰂴𰂵𰂶𰂷𰂸𰂹𰂺𰂻𰂼𰂽𰂾𰂿𰃀𰃁𰃂𰃃𰃄𰃅𰃆𰃇𰃈𰃉𰃊𰃋𰃌𰃍𰃎𰃏𰃐𰃑𰃒𰃓𰃔𰃕𰃖𰃗𰃘𰃙𰃚𰃛𰃜𰃝𰃞𰃟𰃠𰃡𰃢𰃣𰃤𰃥𰃦𰃧𰃨𰃩𰃪𰃫𰃬𰃭𰃮𰃯𰃰𰃱𰃲𰃳𰃴𰃵𰃶𰃷𰃸𰃹𰃺𰃻𰃼𰃽𰃾𰃿𰄀"s, L"𰀀𰀁𰀂𰀃𰀄𰀅𰀆𰀇𰀈𰀉𰀊𰀋𰀌𰀍𰀎𰀏𰀐𰀑𰀒𰀓𰀔𰀕𰀖𰀗𰀘𰀙𰀚𰀛𰀜𰀝𰀞𰀟𰀠𰀡𰀢𰀣𰀤𰀥𰀦𰀧𰀨𰀩𰀪𰀫𰀬𰀭𰀮𰀯𰀰𰀱𰀲𰀳𰀴𰀵𰀶𰀷𰀸𰀹𰀺𰀻𰀼𰀽𰀾𰀿𰁀𰁁𰁂𰁃𰁄𰁅𰁆𰁇𰁈𰁉𰁊𰁋𰁌𰁍𰁎𰁏𰁐𰁑𰁒𰁓𰁔𰁕𰁖𰁗𰁘𰁙𰁚𰁛𰁜𰁝𰁞𰁟𰁠𰁡𰁢𰁣𰁤𰁥𰁦𰁧𰁨𰁩𰁪𰁫𰁬𰁭𰁮𰁯𰁰𰁱𰁲𰁳𰁴𰁵𰁶𰁷𰁸𰁹𰁺𰁻𰁼𰁽𰁾𰁿𰂀𰂁𰂂𰂃𰂄𰂅𰂆𰂇𰂈𰂉𰂊𰂋𰂌𰂍𰂎𰂏𰂐𰂑𰂒𰂓𰂔𰂕𰂖𰂗𰂘𰂙𰂚𰂛𰂜𰂝𰂞𰂟𰂠𰂡𰂢𰂣𰂤𰂥𰂦𰂧𰂨𰂩𰂪𰂫𰂬𰂭𰂮𰂯𰂰𰂱𰂲𰂳𰂴𰂵𰂶𰂷𰂸𰂹𰂺𰂻𰂼𰂽𰂾𰂿𰃀𰃁𰃂𰃃𰃄𰃅𰃆𰃇𰃈𰃉𰃊𰃋𰃌𰃍𰃎𰃏𰃐𰃑𰃒𰃓𰃔𰃕𰃖𰃗𰃘𰃙𰃚𰃛𰃜𰃝𰃞𰃟𰃠𰃡𰃢𰃣𰃤𰃥𰃦𰃧𰃨𰃩𰃪𰃫𰃬𰃭𰃮𰃯𰃰𰃱𰃲𰃳𰃴𰃵𰃶𰃷𰃸𰃹𰃺𰃻𰃼𰃽𰃾𰃿𰄀"s);

    // Supplementary Special-purpose Plane
    allPassed &= test(L"󠀀󠀁󠀂󠀃󠀄󠀅󠀆󠀇󠀈󠀉󠀊󠀋󠀌󠀍󠀎󠀏󠀐󠀑󠀒󠀓󠀔󠀕󠀖󠀗󠀘󠀙󠀚󠀛󠀜󠀝󠀞󠀟󠀠󠀡󠀢󠀣󠀤󠀥󠀦󠀧󠀨󠀩󠀪󠀫󠀬󠀭󠀮󠀯󠀰󠀱󠀲󠀳󠀴󠀵󠀶󠀷󠀸󠀹󠀺󠀻󠀼󠀽󠀾󠀿󠁀󠁁󠁂󠁃󠁄󠁅󠁆󠁇󠁈󠁉󠁊󠁋󠁌󠁍󠁎󠁏󠁐󠁑󠁒󠁓󠁔󠁕󠁖󠁗󠁘󠁙󠁚󠁛󠁜󠁝󠁞󠁟󠁠󠁡󠁢󠁣󠁤󠁥󠁦󠁧󠁨󠁩󠁪󠁫󠁬󠁭󠁮󠁯󠁰󠁱󠁲󠁳󠁴󠁵󠁶󠁷󠁸󠁹󠁺󠁻󠁼󠁽󠁾󠁿"s, u8"󠀀󠀁󠀂󠀃󠀄󠀅󠀆󠀇󠀈󠀉󠀊󠀋󠀌󠀍󠀎󠀏󠀐󠀑󠀒󠀓󠀔󠀕󠀖󠀗󠀘󠀙󠀚󠀛󠀜󠀝󠀞󠀟󠀠󠀡󠀢󠀣󠀤󠀥󠀦󠀧󠀨󠀩󠀪󠀫󠀬󠀭󠀮󠀯󠀰󠀱󠀲󠀳󠀴󠀵󠀶󠀷󠀸󠀹󠀺󠀻󠀼󠀽󠀾󠀿󠁀󠁁󠁂󠁃󠁄󠁅󠁆󠁇󠁈󠁉󠁊󠁋󠁌󠁍󠁎󠁏󠁐󠁑󠁒󠁓󠁔󠁕󠁖󠁗󠁘󠁙󠁚󠁛󠁜󠁝󠁞󠁟󠁠󠁡󠁢󠁣󠁤󠁥󠁦󠁧󠁨󠁩󠁪󠁫󠁬󠁭󠁮󠁯󠁰󠁱󠁲󠁳󠁴󠁵󠁶󠁷󠁸󠁹󠁺󠁻󠁼󠁽󠁾󠁿"s);
    allPassed &= test(u8"󠀀󠀁󠀂󠀃󠀄󠀅󠀆󠀇󠀈󠀉󠀊󠀋󠀌󠀍󠀎󠀏󠀐󠀑󠀒󠀓󠀔󠀕󠀖󠀗󠀘󠀙󠀚󠀛󠀜󠀝󠀞󠀟󠀠󠀡󠀢󠀣󠀤󠀥󠀦󠀧󠀨󠀩󠀪󠀫󠀬󠀭󠀮󠀯󠀰󠀱󠀲󠀳󠀴󠀵󠀶󠀷󠀸󠀹󠀺󠀻󠀼󠀽󠀾󠀿󠁀󠁁󠁂󠁃󠁄󠁅󠁆󠁇󠁈󠁉󠁊󠁋󠁌󠁍󠁎󠁏󠁐󠁑󠁒󠁓󠁔󠁕󠁖󠁗󠁘󠁙󠁚󠁛󠁜󠁝󠁞󠁟󠁠󠁡󠁢󠁣󠁤󠁥󠁦󠁧󠁨󠁩󠁪󠁫󠁬󠁭󠁮󠁯󠁰󠁱󠁲󠁳󠁴󠁵󠁶󠁷󠁸󠁹󠁺󠁻󠁼󠁽󠁾󠁿"s, L"󠀀󠀁󠀂󠀃󠀄󠀅󠀆󠀇󠀈󠀉󠀊󠀋󠀌󠀍󠀎󠀏󠀐󠀑󠀒󠀓󠀔󠀕󠀖󠀗󠀘󠀙󠀚󠀛󠀜󠀝󠀞󠀟󠀠󠀡󠀢󠀣󠀤󠀥󠀦󠀧󠀨󠀩󠀪󠀫󠀬󠀭󠀮󠀯󠀰󠀱󠀲󠀳󠀴󠀵󠀶󠀷󠀸󠀹󠀺󠀻󠀼󠀽󠀾󠀿󠁀󠁁󠁂󠁃󠁄󠁅󠁆󠁇󠁈󠁉󠁊󠁋󠁌󠁍󠁎󠁏󠁐󠁑󠁒󠁓󠁔󠁕󠁖󠁗󠁘󠁙󠁚󠁛󠁜󠁝󠁞󠁟󠁠󠁡󠁢󠁣󠁤󠁥󠁦󠁧󠁨󠁩󠁪󠁫󠁬󠁭󠁮󠁯󠁰󠁱󠁲󠁳󠁴󠁵󠁶󠁷󠁸󠁹󠁺󠁻󠁼󠁽󠁾󠁿"s);
    allPassed &= test(L"󠄀󠄁󠄂󠄃󠄄󠄅󠄆󠄇󠄈󠄉󠄊󠄋󠄌󠄍󠄎󠄏󠄐󠄑󠄒󠄓󠄔󠄕󠄖󠄗󠄘󠄙󠄚󠄛󠄜󠄝󠄞󠄟󠄠󠄡󠄢󠄣󠄤󠄥󠄦󠄧󠄨󠄩󠄪󠄫󠄬󠄭󠄮󠄯󠄰󠄱󠄲󠄳󠄴󠄵󠄶󠄷󠄸󠄹󠄺󠄻󠄼󠄽󠄾󠄿󠅀󠅁󠅂󠅃󠅄󠅅󠅆󠅇󠅈󠅉󠅊󠅋󠅌󠅍󠅎󠅏󠅐󠅑󠅒󠅓󠅔󠅕󠅖󠅗󠅘󠅙󠅚󠅛󠅜󠅝󠅞󠅟󠅠󠅡󠅢󠅣󠅤󠅥󠅦󠅧󠅨󠅩󠅪󠅫󠅬󠅭󠅮󠅯󠅰󠅱󠅲󠅳󠅴󠅵󠅶󠅷󠅸󠅹󠅺󠅻󠅼󠅽󠅾󠅿󠆀󠆁󠆂󠆃󠆄󠆅󠆆󠆇󠆈󠆉󠆊󠆋󠆌󠆍󠆎󠆏󠆐󠆑󠆒󠆓󠆔󠆕󠆖󠆗󠆘󠆙󠆚󠆛󠆜󠆝󠆞󠆟󠆠󠆡󠆢󠆣󠆤󠆥󠆦󠆧󠆨󠆩󠆪󠆫󠆬󠆭󠆮󠆯󠆰󠆱󠆲󠆳󠆴󠆵󠆶󠆷󠆸󠆹󠆺󠆻󠆼󠆽󠆾󠆿󠇀󠇁󠇂󠇃󠇄󠇅󠇆󠇇󠇈󠇉󠇊󠇋󠇌󠇍󠇎󠇏󠇐󠇑󠇒󠇓󠇔󠇕󠇖󠇗󠇘󠇙󠇚󠇛󠇜󠇝󠇞󠇟󠇠󠇡󠇢󠇣󠇤󠇥󠇦󠇧󠇨󠇩󠇪󠇫󠇬󠇭󠇮󠇯"s, u8"󠄀󠄁󠄂󠄃󠄄󠄅󠄆󠄇󠄈󠄉󠄊󠄋󠄌󠄍󠄎󠄏󠄐󠄑󠄒󠄓󠄔󠄕󠄖󠄗󠄘󠄙󠄚󠄛󠄜󠄝󠄞󠄟󠄠󠄡󠄢󠄣󠄤󠄥󠄦󠄧󠄨󠄩󠄪󠄫󠄬󠄭󠄮󠄯󠄰󠄱󠄲󠄳󠄴󠄵󠄶󠄷󠄸󠄹󠄺󠄻󠄼󠄽󠄾󠄿󠅀󠅁󠅂󠅃󠅄󠅅󠅆󠅇󠅈󠅉󠅊󠅋󠅌󠅍󠅎󠅏󠅐󠅑󠅒󠅓󠅔󠅕󠅖󠅗󠅘󠅙󠅚󠅛󠅜󠅝󠅞󠅟󠅠󠅡󠅢󠅣󠅤󠅥󠅦󠅧󠅨󠅩󠅪󠅫󠅬󠅭󠅮󠅯󠅰󠅱󠅲󠅳󠅴󠅵󠅶󠅷󠅸󠅹󠅺󠅻󠅼󠅽󠅾󠅿󠆀󠆁󠆂󠆃󠆄󠆅󠆆󠆇󠆈󠆉󠆊󠆋󠆌󠆍󠆎󠆏󠆐󠆑󠆒󠆓󠆔󠆕󠆖󠆗󠆘󠆙󠆚󠆛󠆜󠆝󠆞󠆟󠆠󠆡󠆢󠆣󠆤󠆥󠆦󠆧󠆨󠆩󠆪󠆫󠆬󠆭󠆮󠆯󠆰󠆱󠆲󠆳󠆴󠆵󠆶󠆷󠆸󠆹󠆺󠆻󠆼󠆽󠆾󠆿󠇀󠇁󠇂󠇃󠇄󠇅󠇆󠇇󠇈󠇉󠇊󠇋󠇌󠇍󠇎󠇏󠇐󠇑󠇒󠇓󠇔󠇕󠇖󠇗󠇘󠇙󠇚󠇛󠇜󠇝󠇞󠇟󠇠󠇡󠇢󠇣󠇤󠇥󠇦󠇧󠇨󠇩󠇪󠇫󠇬󠇭󠇮󠇯"s);
    allPassed &= test(u8"󠄀󠄁󠄂󠄃󠄄󠄅󠄆󠄇󠄈󠄉󠄊󠄋󠄌󠄍󠄎󠄏󠄐󠄑󠄒󠄓󠄔󠄕󠄖󠄗󠄘󠄙󠄚󠄛󠄜󠄝󠄞󠄟󠄠󠄡󠄢󠄣󠄤󠄥󠄦󠄧󠄨󠄩󠄪󠄫󠄬󠄭󠄮󠄯󠄰󠄱󠄲󠄳󠄴󠄵󠄶󠄷󠄸󠄹󠄺󠄻󠄼󠄽󠄾󠄿󠅀󠅁󠅂󠅃󠅄󠅅󠅆󠅇󠅈󠅉󠅊󠅋󠅌󠅍󠅎󠅏󠅐󠅑󠅒󠅓󠅔󠅕󠅖󠅗󠅘󠅙󠅚󠅛󠅜󠅝󠅞󠅟󠅠󠅡󠅢󠅣󠅤󠅥󠅦󠅧󠅨󠅩󠅪󠅫󠅬󠅭󠅮󠅯󠅰󠅱󠅲󠅳󠅴󠅵󠅶󠅷󠅸󠅹󠅺󠅻󠅼󠅽󠅾󠅿󠆀󠆁󠆂󠆃󠆄󠆅󠆆󠆇󠆈󠆉󠆊󠆋󠆌󠆍󠆎󠆏󠆐󠆑󠆒󠆓󠆔󠆕󠆖󠆗󠆘󠆙󠆚󠆛󠆜󠆝󠆞󠆟󠆠󠆡󠆢󠆣󠆤󠆥󠆦󠆧󠆨󠆩󠆪󠆫󠆬󠆭󠆮󠆯󠆰󠆱󠆲󠆳󠆴󠆵󠆶󠆷󠆸󠆹󠆺󠆻󠆼󠆽󠆾󠆿󠇀󠇁󠇂󠇃󠇄󠇅󠇆󠇇󠇈󠇉󠇊󠇋󠇌󠇍󠇎󠇏󠇐󠇑󠇒󠇓󠇔󠇕󠇖󠇗󠇘󠇙󠇚󠇛󠇜󠇝󠇞󠇟󠇠󠇡󠇢󠇣󠇤󠇥󠇦󠇧󠇨󠇩󠇪󠇫󠇬󠇭󠇮󠇯"s, L"󠄀󠄁󠄂󠄃󠄄󠄅󠄆󠄇󠄈󠄉󠄊󠄋󠄌󠄍󠄎󠄏󠄐󠄑󠄒󠄓󠄔󠄕󠄖󠄗󠄘󠄙󠄚󠄛󠄜󠄝󠄞󠄟󠄠󠄡󠄢󠄣󠄤󠄥󠄦󠄧󠄨󠄩󠄪󠄫󠄬󠄭󠄮󠄯󠄰󠄱󠄲󠄳󠄴󠄵󠄶󠄷󠄸󠄹󠄺󠄻󠄼󠄽󠄾󠄿󠅀󠅁󠅂󠅃󠅄󠅅󠅆󠅇󠅈󠅉󠅊󠅋󠅌󠅍󠅎󠅏󠅐󠅑󠅒󠅓󠅔󠅕󠅖󠅗󠅘󠅙󠅚󠅛󠅜󠅝󠅞󠅟󠅠󠅡󠅢󠅣󠅤󠅥󠅦󠅧󠅨󠅩󠅪󠅫󠅬󠅭󠅮󠅯󠅰󠅱󠅲󠅳󠅴󠅵󠅶󠅷󠅸󠅹󠅺󠅻󠅼󠅽󠅾󠅿󠆀󠆁󠆂󠆃󠆄󠆅󠆆󠆇󠆈󠆉󠆊󠆋󠆌󠆍󠆎󠆏󠆐󠆑󠆒󠆓󠆔󠆕󠆖󠆗󠆘󠆙󠆚󠆛󠆜󠆝󠆞󠆟󠆠󠆡󠆢󠆣󠆤󠆥󠆦󠆧󠆨󠆩󠆪󠆫󠆬󠆭󠆮󠆯󠆰󠆱󠆲󠆳󠆴󠆵󠆶󠆷󠆸󠆹󠆺󠆻󠆼󠆽󠆾󠆿󠇀󠇁󠇂󠇃󠇄󠇅󠇆󠇇󠇈󠇉󠇊󠇋󠇌󠇍󠇎󠇏󠇐󠇑󠇒󠇓󠇔󠇕󠇖󠇗󠇘󠇙󠇚󠇛󠇜󠇝󠇞󠇟󠇠󠇡󠇢󠇣󠇤󠇥󠇦󠇧󠇨󠇩󠇪󠇫󠇬󠇭󠇮󠇯"s);

    if (allPassed)
    {
        std::cout << "All tests passed" << std::endl;
        return 0;
    }
    else
    {
        std::cerr << "Some tests failed" << std::endl;
        return 1;
    }
}
