#pragma once
#include <sycl/ext/intel/prototype/internal/_interfaces_annotations.hpp>

// This file contains the following sections:
//   Section 1. Usage Guide for kernel control interface macros.
//   Section 2. Usage Guide for kernel argument interface macros.
//   Section 3. Usage Guide for the buffer_location macro.
//
// The first two section are useful for defining interfaces of a FPGA kernel
// when creating a library IP file.
//
// Section 3 describes the 'buffer_location' macro which should be used for
// assigning a buffer location to a USM pointer either when compiling a program
// for a specific FPGA board or when creating a library IP file.

// Section 1. Usage Guide for kernel control interface macros:
// #############################################################################
// The macros defined in this section can be used to specify the following
// kernel control interface settings:
//
//   1. Streaming Interface
//   2. Register Map Interface
//   3. Pipelined Interface
//   4. Pipelined Streaming Interface
//   5. Pipelined Register Map Interface
//
// NOTE:
// - If Streaming kernel control interface is specified, the kernel argument
//   interfaces become 'conduits'.
// - If Register Map based kernel control interface is chosen, the kernel
//   arguments become 'register_map' based interfaces.
// - See Section 2 below to see how to explicitly set kernel argument interfaces
//
// Please copy one of the macros that include all the settings desired. An
// example is shown below.
//
// Example:
// ########
// For an IP that will have streaming kernel control interface and will be
// pipelined, use the following:

// struct MyIP {
//   ...
//   streaming_pipelined_interface
//   void operator()() {
//     // device code
//     ...
//   }
// };

// Streaming Interface
#define streaming_interface _streaming_interface_annotation

// Register Map Interface
#define register_map_interface _register_map_interface_annotation

// Pipelined Interface
#define pipelined_interface _pipelined_interface_annotation

// Pipelined Streaming Interface
#define streaming_pipelined_interface _streaming_pipelined_interface_annotation

// Pipelined Register Map Interface
#define register_map_pipelined_interface _register_map_pipelined_interface_annotation

// Section 2. Usage Guide for kernel argument interface macros:
// #############################################################################
// The macros defined in this section can be used to generate the following
// types of kernel argument interfaces:
//   1. register_map
//   2. conduit
//
//  NOTE:
//  - Customized mm_host arguments can be specified with the mmhost macros.
//    Please copy the macro templates shown below into your source file and fill
//    in the values for different parameters as needed.

// Examples:
// #########

// 1. Register Map Argument
// #############################################################################
// struct MyIp {
//   register_map int input_a;
//   ...
// };

// 2. Conduit Argument
// #############################################################################
// struct MyIp {
//   conduit int input_a;
//   ...
// };

// 3. MM Host Argument
// #############################################################################
// The pointer address will be input via a conduit if 'streaming_interface'
// macro is specifed. A read only input register will be store the address
// otherwise.

// Example below fills in the default values

// struct MyIp {
//   mmhost(
//     1,   // buffer_location
//     28,  // address width
//     64,  // data width
//     16,  // latency
//     0,   // read_write_mode, 0: ReadWrite, 1: Read, 2: Write
//     1,   // maxburst
//     0,   // align, 0 defaults to alignment of the type
//     0    // enable waitrequest, 0: no, 1: yes
//   ) int *input_a;
//   ...
// };

// 4. Conduit Input for MM Host Kernel Argument
// #############################################################################
// Example below fills in the default values

// struct MyIp {
//   conduit_mmhost(
//     1,   // buffer_location
//     28,  // address width
//     64,  // data width
//     16,  // latency, must be atleast 16
//     0,   // read_write_mode, 0: ReadWrite, 1: Read, 2: Write
//     1,   // maxburst
//     0,   // align, 0 defaults to alignment of the type
//     0    // enable waitrequest, 0: no, 1: yes
//   ) int *input_a;
//   ...
// };

// 5. Register Map based Input for MM Host Kernel Argument
// #############################################################################
// Example below fills in the default values

// struct MyIp {
//   register_map_mmhost(
//     1,   // buffer_location
//     28,  // address width
//     64,  // data width
//     16,  // latency, must be atleast 16
//     0,   // read_write_mode, 0: ReadWrite, 1: Read, 2: Write
//     1,   // maxburst
//     0,   // align, 0 defaults to alignment of the type
//     0    // enable waitrequest, 0: no, 1: yes
//   ) int *input_a;
//   ...
// };

// MM Host Macro
#define mmhost(                                                                \
  buffer_location,                                                             \
  awidth,                                                                      \
  dwidth,                                                                      \
  latency,                                                                     \
  read_write_mode,                                                             \
  maxburst,                                                                    \
  align,                                                                       \
  waitrequest) __mmhost_annotation(buffer_location, awidth, dwidth,            \
    latency, read_write_mode, maxburst, align, waitrequest)

// Conduit Argument Macro for MM Host Base Address
#define conduit_mmhost(                                                        \
  buffer_location,                                                             \
  awidth,                                                                      \
  dwidth,                                                                      \
  latency,                                                                     \
  read_write_mode,                                                             \
  maxburst,                                                                    \
  align,                                                                       \
  waitrequest) __conduit_mmhost_annotation(buffer_location, awidth,            \
    dwidth, latency, read_write_mode, maxburst, align, waitrequest)

// Register Map Argument Macro for MM Host Base Address
#define register_map_mmhost(                                                   \
  buffer_location,                                                             \
  awidth,                                                                      \
  dwidth,                                                                      \
  latency,                                                                     \
  read_write_mode,                                                             \
  maxburst,                                                                    \
  align,                                                                       \
  waitrequest) __register_map_mmhost_annotation(buffer_location, awidth,       \
    dwidth, latency, read_write_mode, maxburst, align, waitrequest)

// Register Map Argument Macro
#define register_map __register_map_annotation

// Conduit Argument Macro
#define conduit __conduit_annotation

// Conduit Stable Argument Macro
#define stable_conduit [[clang::annotate("__UserSemINTEL__ conduit_annotation", 1)]]

// Section 3. Usage Guide for the buffer_location macro:
// #############################################################################
// The macro defined in this section can be used to specify a 'buffer_location'
// value for the kernel's pointer arguments.
//
// NOTE: - The macro only works with a kernel functor and not with a kernel
//         lambda. Please use it as shown below.
//       - When creating an IP file, the pointer argument will receive the
//         default mm_host parameters and the argument interface will be the
//         same as the kernel control interface.
//
#define ptr_buffer_location(value) __buffer_location_annotation(value)

// Usage Example:
// ##############
// struct MyKernel {
//   ptr_buffer_location(1) int *input_a;
//   ptr_buffer_location(2) int *input_b;
//   ptr_buffer_location(3) int *input_c;
//   ...
// };

